package com.lanren.huhu.partner.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.lanren.huhu.partner.dao.PartnerAccountMapper;
import com.lanren.huhu.partner.dao.PartnerInviteRelationMapper;
import com.lanren.huhu.partner.dao.UserAgentMapper;
import com.lanren.huhu.partner.domain.PartnerAccount;
import com.lanren.huhu.partner.domain.PartnerInviteRelation;
import com.lanren.huhu.partner.domain.UserAgent;
import com.lanren.huhu.partner.model.ParentAgent;
import com.lanren.huhu.partner.model.ParentPartner;
import com.lanren.huhu.partner.model.User;
import com.lanren.huhu.partner.service.UserService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import static com.lanren.huhu.partner.constants.Constants.PARENT_COLUMN_NAME_CENGJI;
import static com.lanren.huhu.partner.constants.Constants.PARENT_COLUMN_NAME_PINGTUI;

/**
 * @author houseme
 * @date 2019-06-28 18:36
 * @Project partner
 * @Package com.lanren.huhu.partner.service.impl
 * @File: UserServiceImpl
 */
@Service
@CacheConfig(cacheNames = "user:relation:cache")
public class UserServiceImpl implements UserService {
    private static Logger logger = LoggerFactory.getLogger(UserServiceImpl.class);

    @Autowired
    private UserAgentMapper userAgentMapper;
    @Autowired
    private PartnerAccountMapper partnerAccountMapper;
    @Autowired
    private PartnerInviteRelationMapper partnerInviteRelationMapper;


    /**
     * ****************************************分割线****用户邀请关系***************************************************************
     */

    /**
     * 用户上级 合伙人 和 代理 关系
     * @param userId
     * @return
     */
    @Override
    @Cacheable
    public User getRelationByUserId(Integer userId) {
        logger.info("UserServiceImpl get user relation from database, userId: {}", userId);
        User user = new User();
        user.setUserId(userId);
        user.setPartnerList((ArrayList<ParentPartner>) getPartnerListByUserId(userId));
        user.setAgentList((ArrayList<ParentAgent>) getAgentListByUserId(userId, PARENT_COLUMN_NAME_CENGJI));
        user.setPingtuiAgentList((ArrayList<ParentAgent>) getAgentListByUserId(userId, PARENT_COLUMN_NAME_PINGTUI));
        return user;
    }

    @Override
    @CacheEvict(key = "#userId")
    public boolean deleteRalationByUserId(Integer userId) {
        logger.info("UserServiceImpl delete user relation cache, userId: {}", userId);
        return true;
    }

    /**
     * ****************************************分割线****合伙人***************************************************************
     */

    /**
     * 合伙人数据
     * @param userId
     * @return
     */
    @Override
    public PartnerAccount getPartnerAccountByUserId(Integer userId) {
        QueryWrapper<PartnerAccount> queryWrapper = new QueryWrapper<PartnerAccount>();
        queryWrapper.eq("user_id", userId);
        return partnerAccountMapper.selectOne(queryWrapper);
    }
    @Override
    public List<PartnerAccount> getPartnerAccountChildListByUserId(Integer userId) {
        return partnerAccountMapper.getChildListByUserId(userId);
    }

    /**
     * 按userId查找上级合伙人的 userId 和 level
     * @param userId
     * @return
     */
    private ParentPartner getParentPartner(Integer userId) {
        PartnerAccount partnerAccount = partnerAccountMapper.getParentByUserId(userId);
        ParentPartner parentPartner = null;
        if (null != partnerAccount) {
            int partnerLevel = partnerAccount.getIsSuperPartner() == 1 ? 30 : partnerAccount.getPartnerLevel();
            parentPartner = new ParentPartner(partnerAccount.getUserId(),partnerLevel);
        }
        return parentPartner;
    }

    /**
     * @description: 查找一个用户的全部合伙人上级
     * @param userId
     * @author chen
     * @return List<ParentPartner>
     */
    private List<ParentPartner> getPartnerListByUserId(Integer userId) {
        if (logger.isInfoEnabled()) {
            logger.info("UserServiceImpl getPartnerListByUserId for user: {} ......", userId);
        }
        ArrayList<ParentPartner> partnerList = new ArrayList<ParentPartner>();
        /**
         * 无限找
         */
        int loopUserId = userId;
        int loopTimes = 0;
        while (true) {
            if (logger.isDebugEnabled()) {
                logger.info("now loop uid is {}", loopUserId);
            }
            ParentPartner parentPartner = getParentPartner(loopUserId);
            if (null == parentPartner) {
                break;
            } else {
                if (logger.isDebugEnabled()) {
                    logger.info("parent is {} ", parentPartner.getUserId());
                }
                if (partnerList.contains(parentPartner) || parentPartner.getUserId() == userId) {
                    logger.info("发现互相邀请的关系 用户({})", parentPartner.getUserId());
                    logger.info("退出循环, 不再继续查找");
                    break;
                }
                partnerList.add(parentPartner);
                loopUserId = parentPartner.getUserId();
                loopTimes++;
                if (loopTimes > 500) {
                    partnerList.clear();
                    logger.error("用户:{} 邀请关系异常, 已超500层", userId);
                    break;
                }
            }
        }
        if (logger.isInfoEnabled()) {
            logger.info("setPartnerList for user: {} end", userId);
        }
        return partnerList;
    }

    @Override
    public int updatePartnerRelation(String from, String to) {
        PartnerAccount fromPartnerAccount = partnerAccountMapper.getPartnerAccountByPhone(from);
        PartnerAccount toPartnerAccount = partnerAccountMapper.getPartnerAccountByPhone(to);
        if (null != fromPartnerAccount && null != toPartnerAccount) {
            QueryWrapper<PartnerInviteRelation> wrapper = new QueryWrapper<PartnerInviteRelation>();
            wrapper.eq("user_id", fromPartnerAccount.getUserId());
            PartnerInviteRelation partnerInviteRelation = partnerInviteRelationMapper.selectOne(wrapper);
            partnerInviteRelation.setInviteUserId(toPartnerAccount.getUserId());
            partnerInviteRelation.setInvitePartnerLevel(toPartnerAccount.getIsSuperPartner() == 1 ? 30 : toPartnerAccount.getPartnerLevel());
            return partnerInviteRelationMapper.updateById(partnerInviteRelation);
        } else {
            return 0;
        }
    }

    @Override
    public PartnerAccount getPartnerAccountByPhone(String phone) {
        return partnerAccountMapper.getPartnerAccountByPhone(phone);
    }


    /**
     * ****************************************分割线****代理商***************************************************************
     */

    /**
     * 代理商数据
     * @param userId
     * @return
     */
    @Override
    public UserAgent getUserAgentByUserId(Integer userId) {
        QueryWrapper<UserAgent> queryWrapper = new QueryWrapper<UserAgent>();
        queryWrapper.eq("user_id", userId).le("agent_level", 4);
        return userAgentMapper.selectOne(queryWrapper);
    }

    @Override
    public List<UserAgent> getUserAgentChildListByUserId(Integer userId, String parentColumnName) {
        return userAgentMapper.getChildListByUserId(userId, parentColumnName);
    }

    @Override
    public int updatePingtuiRelation(String from, String to) {
        UserAgent fromUa = userAgentMapper.getUserAgentByPhone(from);
        UserAgent toUa = userAgentMapper.getUserAgentByPhone(to);
        if (null != fromUa && null != toUa) {
            fromUa.setPresenterId(toUa.getAgentId());
        }
        return userAgentMapper.updateById(fromUa);
    }

    @Override
    public int updateCengjiRelation(String from, String to) {
        UserAgent fromUa = userAgentMapper.getUserAgentByPhone(from);
        UserAgent toUa = userAgentMapper.getUserAgentByPhone(to);
        if (null != fromUa && null != toUa) {
            fromUa.setParentAgentId(toUa.getAgentId());
        }
        return userAgentMapper.updateById(fromUa);
    }

    /**
     * 按agentId 查找上级代理商的 agentId 和 level
     * @param agentId
     * @return
     */
    private ParentAgent getParentAgent(Integer agentId, String parentColumnName) {
        UserAgent userAgent = userAgentMapper.getParentAgent(agentId, parentColumnName);
        ParentAgent parentAgent = null;
        if (null != userAgent) {
            int agentLevel = userAgent.getAgentLevel();
            parentAgent = new ParentAgent(userAgent.getUserId(), userAgent.getAgentId(), agentLevel);
        }
        return parentAgent;
    }

    /**
     * @description: 查找一个用户的全部代理商上级
     * 注意 !!!!!!!!  这个方法是按输入用户的邀请关系链上查找 第一个有agent_level<=4的代理商身份的邀请人, 然后再走他的代理关系
     * 注意 !!!!!!!!  如果需要获取一个代理商的上级代理 需要从UserAgentService里的方法获取
     * @param userId
     * @return List<ParentAgent>
     */
    @Override
    public List<ParentAgent> getAgentListByUserId(Integer userId, String parentColumnName) {
        if (logger.isInfoEnabled()) {
            logger.info("getAgentListByUserId for user: {} ......", userId);
        }
        ArrayList<ParentAgent> agentList = new ArrayList<ParentAgent>();
        HashSet<Integer> existingUserIdSet = new HashSet<Integer>();
        /**
         * 无限找
         * 先无限找上级合伙人, 直到找到第一个是代理商的合伙人
         */

        /**
         * 要改成从自己开始找 所以添加判断 看自己是不是
         */
        UserAgent selfAgent = getUserAgentByUserId(userId);
        int loopUserId = userId;
        int loopTimes = 0;
        while (true) {
            if (logger.isDebugEnabled()) {
                logger.info("now loop uid is {}", loopUserId);
            }
            existingUserIdSet.add(loopUserId);
            ParentPartner parentPartner = getParentPartner(loopUserId);
            /**
             * 有上级合伙人  或者  自己就是代理商
             */
            if (null != selfAgent || null != parentPartner) {
                UserAgent firstAgent;
                if (null != selfAgent) {
                    firstAgent = selfAgent;
                } else {
                    firstAgent = getUserAgentByUserId(parentPartner.getUserId());
                }
                /**
                 * 找到了第一个是代理商的合伙人
                 */
                if (null != firstAgent) {
                    /**
                     * 然后开始 无限找 这个人的上级代理商
                     */
                    if (logger.isDebugEnabled()) {
                        logger.info("find first agent({}) for user: {}", firstAgent.getAgentId(), userId);
                    }
                    agentList.add(new ParentAgent(firstAgent.getUserId(), firstAgent.getAgentId(), firstAgent.getAgentLevel()));
                    loopTimes++;
                    if (logger.isDebugEnabled()) {
                        logger.info("loop find parent agent start ......");
                    }
                    int loogAgentId = firstAgent.getAgentId();
                    while (true) {
                        ParentAgent loopUserAgent = getParentAgent(loogAgentId, parentColumnName);
                        if (null == loopUserAgent) {
                            break;
                        } else {
                            if (logger.isDebugEnabled()) {
                                logger.info("parent agent is ", loopUserAgent.getAgentId());
                            }
                            if (agentList.contains(loopUserAgent) || loopUserAgent.getAgentId() == firstAgent.getAgentId()) {
                                logger.info("发现互为上级的代理关系 代理商({})", loopUserAgent.getAgentId());
                                logger.info("退出循环, 不再继续查找");
                                break;
                            }
                            agentList.add(new ParentAgent(loopUserAgent.getUserId(), loopUserAgent.getAgentId(), loopUserAgent.getLevel()));
                            loogAgentId = loopUserAgent.getAgentId();
                            loopTimes++;
                            if (loopTimes > 500) {
                                agentList.clear();
                                logger.error("用户ID:{} 代理商关系异常, 已超500层", userId);
                                break;
                            }
                        }
                    }
                    /**
                     * 找完 "第一个是代理商的合伙人" 的 全部上级代理后 退出while循环
                     */
                    if (logger.isDebugEnabled()) {
                        logger.info("loop find parent agent end");
                    }
                    break;
                } else {
                    /**
                     * 这个人不是代理商 继续往上找
                     */
                    loopUserId = parentPartner.getUserId();
                    if (existingUserIdSet.contains(loopUserId)) {
                        logger.info("发现互为上级的用户邀请关系 用户ID({})", loopUserId);
                        logger.info("退出循环, 不再继续查找");
                        break;
                    }
                }
            } else {
                /**
                 * 没有上级合伙人 打印结束信息 跳出循环
                 */
                if (logger.isDebugEnabled()) {
                    logger.info("loop end with userId: {}", loopUserId);
                }
                break;
            }
        }
        if (logger.isInfoEnabled()) {
            logger.info("setAgentList for user: {} end", userId);
        }
        return agentList;
    }
}
