package com.lanren.huhu.partner.schedule;

import com.lanren.huhu.partner.domain.AgentRevokeRecords;
import com.lanren.huhu.partner.domain.AgentReward;
import com.lanren.huhu.partner.service.AgentRevokeRecordsService;
import com.lanren.huhu.partner.service.AgentRewardService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.List;

import static com.lanren.huhu.partner.constants.Constants.*;

/**
 * @author chen
 * @title: RevokeAgentRewardTask
 * @projectName partner
 * @description: 扣回代理商的奖励
 * @package com.lanren.huhu.partner.schedule
 * @date 2019-07-04 11:41
 */
@Component
public class RevokeAgentRewardTask {
    private static Logger logger = LoggerFactory.getLogger(RevokeAgentRewardTask.class);

    @Autowired
    AgentRevokeRecordsService agentRevokeRecordsService;
    @Autowired
    AgentRewardService agentRewardService;

    @Scheduled(fixedDelay = 5000L)
    public void runScheduledTask() {
        logger.info("run RevokeAgentRewardTask");
        runConsume();
    }

    private void runConsume() {
        List<AgentRevokeRecords> revokeRecordList = null;
        while (true) {
            try {
                revokeRecordList = agentRevokeRecordsService.getAllPending();
                if (null == revokeRecordList || revokeRecordList.size() == 0) {
                    break;
                }
                process(revokeRecordList);
            } catch (Exception e) {
                logger.error("process revoke failed. {}", revokeRecordList);
                logger.error(e.getMessage(), e);
            }
        }
    }

    @Async
    public void process(List<AgentRevokeRecords> revokeRecordList) {
        for (AgentRevokeRecords agentRevokeRecord : revokeRecordList) {
            process(agentRevokeRecord);
        }
    }

    public void process(AgentRevokeRecords revokeRecord) {
        int revokeType = revokeRecord.getRevokeType();
        if (!AGENT_REVOKE_TYPE_MAP.keySet().contains(revokeType)) {
            logger.info("未知的奖励类型 跳过不处理: {}", revokeRecord);
            return;
        }
        AgentReward agentReward = null;
        if (revokeType == AGENT_REVOKE_TYPE_ACTIVATION || revokeType == AGENT_REVOKE_TYPE_SELF_ORDER) {
            agentReward = agentRewardService.getOneByRefId(revokeRecord.getRefId());
        } else if (revokeType == AGENT_REVOKE_TYPE_SHARE_ORDER) {
            agentReward = agentRewardService.getOneBySubOrderSn(revokeRecord.getOrderType(), revokeRecord.getOrderSn(), revokeRecord.getSubOrderSn());
        } else {
            revokeRecord.setState(REVOKE_STATE_DO_NOTHING);
        }
        /**
         * 已结算状态 插负, 未结算状态 抹平
         */
        if (null != agentReward) {
            if (agentReward.getSettleState() == SETTLE_STATE_DONE) {
                agentReward.setAmount(agentReward.getAmount().negate());
                agentReward.setCommission(agentReward.getCommission().negate());
                if (agentRewardService.insertOne(agentReward) > 0) {
                    revokeRecord.setState(REVOKE_STATE_DONE);
                } else {
                    revokeRecord.setState(REVOKE_STATE_FAILED);
                }
            } else if (agentReward.getSettleState() == SETTLE_STATE_ON_THE_WAY || agentReward.getSettleState() == SETTLE_STATE_PRE) {
                agentReward.setAmount(new BigDecimal(0));
                agentReward.setCommission(new BigDecimal(0));
                if (agentRewardService.updateById(agentReward)) {
                    revokeRecord.setState(REVOKE_STATE_DONE);
                } else {
                    revokeRecord.setState(REVOKE_STATE_FAILED);
                }
            } else {
                revokeRecord.setState(REVOKE_STATE_DO_NOTHING);
            }
        } else {
            revokeRecord.setState(REVOKE_STATE_DO_NOTHING);
        }
        agentRevokeRecordsService.updateById(revokeRecord);
    }
}
