package com.lanren.huhu.partner.schedule;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.lanren.huhu.partner.constants.Constants;
import com.lanren.huhu.partner.domain.*;
import com.lanren.huhu.partner.model.PartnerSettle;
import com.lanren.huhu.partner.model.PartnerSnapshot;
import com.lanren.huhu.partner.service.*;
import com.lanren.huhu.partner.util.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.stereotype.Component;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;

import static org.springframework.util.StringUtils.isEmpty;

/**
 * @author chen
 * @title: PartnerSettleTask
 * @projectName partner
 * @description: 合伙人结算
 * @package com.lanren.huhu.partner.schedule
 * @date 2019-07-02 15:45
 */
@Component
public class PartnerSettleTask {
    private static Logger logger = LoggerFactory.getLogger(PartnerSettleTask.class);
    private static final int PAGE_SIZE = 1000;
    private static final BigDecimal ZERO = new BigDecimal(0);

    @Autowired
    PartnerRewardService partnerRewardService;
    @Autowired
    OrdersActiveService ordersActiveService;
    @Autowired
    OrderTaobaoJdCommissionService orderTaobaoJdCommissionService;
    @Autowired
    OrderTaobaoJdGoodsCommissionService orderTaobaoJdGoodsCommissionService;
    @Autowired
    DataSourceTransactionManager dataSourceTransactionManager;
    @Autowired
    TransactionDefinition transactionDefinition;
    @Autowired
    PartnerAccountService partnerAccountService;
    @Autowired
    PartnerAccountLogService partnerAccountLogService;
    @Autowired
    PartnerIncomeSummaryService partnerIncomeSummaryService;

//    @Scheduled(fixedDelay = 30 * 24 * 60 * 60 * 1000L)
    public void runScheduledTask() {
        logger.info("run PartnerSettleTask");
        try {
            long input = DateUtils.getLastMonth1stTimestamp(System.currentTimeMillis());
            long beginTs = DateUtils.getMonth1stTimestamp(input) / 1000L;
            long endTs  = DateUtils.getMonthLastTimestamp(input) / 1000L;
            updateAll(beginTs, endTs);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
    }

    private void updateAll(long beginTs, long endTs) {
        updateOrderCommission(beginTs, endTs, 0, 0);
        doSettle(beginTs, endTs);
    }

    public boolean doOneUserSettle(Integer userId) {
        long input = DateUtils.getLastMonth1stTimestamp(System.currentTimeMillis());
        long beginTs = DateUtils.getMonth1stTimestamp(input) / 1000L;
        long endTs  = DateUtils.getMonthLastTimestamp(input) / 1000L;
        PartnerSettle errPartnerSettle = null;
        try {
            updateOrderCommission(beginTs, endTs, 1, userId);
            if (partnerRewardService == null) {
                logger.info("partnerRewardService is nuull");
            }
            PartnerSettle partnerSettle = partnerRewardService.selectPartnerSettlePageByTsAndUid(beginTs, endTs, userId);
            errPartnerSettle = partnerSettle;
            if (setIncomeAndTax(partnerSettle)) {
                return doDbUpdate(partnerSettle, beginTs, endTs);
            } else {
                return false;
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            logger.error("userId is {}, errPartnerSettle is {}", userId, errPartnerSettle);
            return false;
        }
    }

    private void doSettle(long beginTs, long endTs) {
        PartnerSettle errPartnerSettle = null;
        try {
            int pageNo = 1;
            while (true) {
                Page<PartnerSettle> page = new Page<PartnerSettle>(pageNo, PAGE_SIZE);
                page.setOptimizeCountSql(false);
                page.setSearchCount(false);
                IPage<PartnerSettle> ipage = partnerRewardService.selectPartnerSettlePageByTs(page, beginTs, endTs);
                List<PartnerSettle> partnerSettleList = ipage.getRecords();
                if (partnerSettleList != null && partnerSettleList.size() > 0) {
                    for (PartnerSettle partnerSettle : partnerSettleList) {
                        errPartnerSettle = partnerSettle;
                        if (setIncomeAndTax(partnerSettle)) {
                            if (!doDbUpdate(partnerSettle, beginTs, endTs)) {
                                logger.error("doDbUpdate failed for user {}", partnerSettle.getUserId());
                            }
                        }
                    }
                    pageNo++;
                } else {
                    break;
                }
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            logger.error("errPartnerSettle is {}", errPartnerSettle);
        }
    }

    private boolean doDbUpdate(PartnerSettle partnerSettle, long beginTs, long endTs) {
        TransactionStatus transactionStatus = null;
        try {
            logger.info("partnerSettle is {}", partnerSettle);
            /**
             * 开启事务
             */
            transactionStatus = dataSourceTransactionManager.getTransaction(transactionDefinition);

            /**
             * 获取合伙人账户 锁定行
             */
            PartnerAccount partnerAccount = partnerAccountService.getOneByUserIdForUpdate(partnerSettle.getUserId());
            /**
             * 插入变更记录
             */
            insertIntoPartnerAccountLog(partnerSettle, partnerAccount);
            updatePartnerAccount(partnerSettle, partnerAccount);
            updatePartnerRewardStatus(partnerSettle.getUserId(), beginTs, endTs);
            insertPartnerIncomeSummary(partnerSettle.getUserId(), beginTs);
            /**
             * 提交事务
             */
            dataSourceTransactionManager.commit(transactionStatus);
//            dataSourceTransactionManager.rollback(transactionStatus);
            return true;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            /**
             * 回滚事务
             */
            if (transactionStatus != null) {
                logger.info("do settle failed, rollback");
                dataSourceTransactionManager.rollback(transactionStatus);
            }
            return false;
        }
    }

    private void insertPartnerIncomeSummary(int userId, long beginTs) {
        Date now = new Date(beginTs * 1000L);
        int yeartime = Integer.parseInt(DateUtils.getYear(now));
        int monthtime = Integer.parseInt(DateUtils.getMonth(now));
        PartnerIncomeSummary partnerIncomeSummary = partnerIncomeSummaryService.getOneByUidYearMonth(userId, yeartime, monthtime);
        if (null == partnerIncomeSummary) {
            partnerIncomeSummary = new PartnerIncomeSummary();
            partnerIncomeSummary.setUserId(userId);
            partnerIncomeSummary.setYeartime(yeartime);
            partnerIncomeSummary.setMonthtime(monthtime);
            partnerIncomeSummary.setIncome(ZERO);
            partnerIncomeSummary.setState(15);
            partnerIncomeSummaryService.save(partnerIncomeSummary);
        } else {
            partnerIncomeSummary.setState(15);
            partnerIncomeSummaryService.updateById(partnerIncomeSummary);
        }
    }

    private void updatePartnerRewardStatus(int userId, long beginTs, long endTs) {
        partnerRewardService.updateRewardStatusByUidAndTs(userId, beginTs, endTs);
    }

    private void updatePartnerAccount(PartnerSettle partnerSettle, PartnerAccount partnerAccount) {
        /**
         * 更新合伙人账户金额
         */
        BigDecimal wxIncomeAdd = partnerSettle.getWxIncome().subtract(partnerSettle.getWxTax());
        BigDecimal bankIncomeAdd = partnerSettle.getBankIncome().subtract(partnerSettle.getBankTax());
        partnerAccount.setAllIncome(partnerAccount.getAllIncome().add(wxIncomeAdd).add(bankIncomeAdd));
        partnerAccount.setIncomeAfterTax(partnerAccount.getIncomeAfterTax().add(wxIncomeAdd).add(bankIncomeAdd));
        partnerAccount.setWxFreeBalance(partnerAccount.getWxFreeBalance().add(wxIncomeAdd));
        partnerAccount.setBankFreeBalance(partnerAccount.getBankFreeBalance().add(bankIncomeAdd));
        partnerAccountService.updateById(partnerAccount);
    }

    private void insertIntoPartnerAccountLog(PartnerSettle partnerSettle, PartnerAccount partnerAccount) {
        /**
         * 月结发钱
         */
        BigDecimal changeNum = partnerSettle.getSumOrderCommission().add(partnerSettle.getSumRedpack()).add(partnerSettle.getSumOthers()).add(partnerSettle.getTechCharge());
        BigDecimal changedAmount = partnerAccount.getWxFreeBalance().add(partnerAccount.getBankFreeBalance()).add(changeNum);
        PartnerSnapshot snapshot = new PartnerSnapshot();
        snapshot.setUserId(partnerAccount.getUserId());
        snapshot.setWxFreeBalance(partnerAccount.getWxFreeBalance());
        snapshot.setBankFreeBalance(partnerAccount.getBankFreeBalance());
        PartnerAccountLog log = createNewPartnerAccountLog(partnerSettle.getUserId(),
                Constants.PARTNER_ACCOUNT_LOG_TYPE_SETTLE,
                changeNum,
                changedAmount,
                Constants.PARTNER_ACCOUNT_LOG_TYPE_SETTLE_REMARK,
                snapshot.toString());
        partnerAccountLogService.save(log);
        /**
         * 平台技术服务费
         */
        changeNum = partnerSettle.getTechCharge().negate();
        changedAmount = changedAmount.add(changeNum);
        log = createNewPartnerAccountLog(partnerSettle.getUserId(),
                Constants.PARTNER_ACCOUNT_LOG_TYPE_TECH_CHARGE,
                changeNum,
                changedAmount,
                Constants.PARTNER_ACCOUNT_LOG_TYPE_TECH_CHARGE_REMARK,
                "");
        partnerAccountLogService.save(log);
        /**
         * 个人所得税
         */
        changeNum = partnerSettle.getWxTax().add(partnerSettle.getBankTax()).negate();
        changedAmount = changedAmount.add(changeNum);
        snapshot.setWxFreeBalance(snapshot.getWxFreeBalance().add(partnerSettle.getWxIncome()));
        snapshot.setBankFreeBalance(snapshot.getBankFreeBalance().add(partnerSettle.getBankIncome()));
        log = createNewPartnerAccountLog(partnerSettle.getUserId(),
                Constants.PARTNER_ACCOUNT_LOG_TYPE_TAX,
                changeNum,
                changedAmount,
                Constants.PARTNER_ACCOUNT_LOG_TYPE_TAX_REMARK,
                snapshot.toString());
        partnerAccountLogService.save(log);
    }

    private PartnerAccountLog createNewPartnerAccountLog(int userId, int type, BigDecimal changeNum, BigDecimal changedAmount, String payRemark,String snapshot) {
        Date now = new Date();
        int yeartime = Integer.parseInt(DateUtils.getYear(now));
        int monthtime = Integer.parseInt(DateUtils.getMonth(now));
        int daytime = Integer.parseInt(DateUtils.getDay(now));
        PartnerAccountLog log = new PartnerAccountLog();
        log.setUserId(userId);
        log.setType(type);
        log.setChangeNum(changeNum);
        log.setChangedAmount(changedAmount);
        log.setPayRemark(payRemark);
        log.setPayType(false);
        log.setYeartime(yeartime);
        log.setMonthtime(monthtime);
        log.setDaytime(daytime);
        log.setSnapshot(snapshot.toString());
        return log;
    }

    private void updateRewardCommission(PartnerReward partnerReward, BigDecimal allMoney) {
        partnerReward.setAllMoneyOri(allMoney);
        String ratioAllStr = partnerReward.getRatioAll().replaceAll("[\\[\\],\"]", "");
        String[] strs = ratioAllStr.split(",");
        BigDecimal cash = allMoney;
        for (String s : strs) {
            BigDecimal ratio = new BigDecimal(s);
            cash = cash.multiply(ratio);
        }
        BigDecimal commission = cash.multiply(partnerReward.getCommissionRate());
        partnerReward.setCash(cash);
        partnerReward.setCommissionAcount(commission);
        partnerReward.setSettleState(Constants.SETTLE_STATE_DONE);
        partnerRewardService.updateById(partnerReward);
    }
    /**
     * 更新至 失效, 结算完成
     * 不再参与后续结算计算
     * @param partnerReward
     */
    private void updateReward2Failed(PartnerReward partnerReward) {
        partnerReward.setAllMoneyOri(ZERO);
        partnerReward.setCash(ZERO);
        partnerReward.setCommissionAcount(ZERO);
        partnerReward.setSettleState(Constants.SETTLE_STATE_FAILED);
        partnerReward.setRewardStatus(Constants.REWARD_STATUS_SETTLE);
        partnerRewardService.updateById(partnerReward);
    }

    /**
     * 自购省和分享赚的奖励金额 需要根据最终的实际原始佣金 重新计算
     * @param beginTs
     * @param endTs
     */
    private void updateOrderCommission(long beginTs, long endTs, int runType, int userId) {
        PartnerReward errPartnerReward = null;
        try {
            int pageNo = 1;
            while (true) {
                Page<PartnerReward> page = new Page<PartnerReward>(pageNo, PAGE_SIZE);
                page.setOptimizeCountSql(false);
                page.setSearchCount(false);
                IPage<PartnerReward> ipage;
                if (runType == 0) {
                    ipage = partnerRewardService.selectOrderRewardPageByTs(page, beginTs, endTs);
                } else {
                    ipage = partnerRewardService.selectOrderRewardPageByTsAndUid(page, beginTs, endTs, userId);
                }
                List<PartnerReward> partnerRewardList = ipage.getRecords();
                if (partnerRewardList != null && partnerRewardList.size() > 0) {
                    /**
                     * 循环处理每条奖励
                     */
                    for (PartnerReward partnerReward : partnerRewardList) {
                        errPartnerReward = partnerReward;
                        /**
                         * 购物奖励: reward_type: 110, settle_state > 0 且 != 300
                         * 需要根据订单号查找最终的真实的原始佣金 并 按比例快照重新计算奖励金额,
                         * 更新字段: all_money_ori cash commission_account
                         * 如果订单失效: 更新settle_state, all_money_ori/cash/commission_account 抹平, 跳过处理, 循环到下一条
                         */
                        if (partnerReward.getRewardType() == Constants.PARTNER_REWARD_TYPE_PURCHASE && partnerReward.getSettleState() > 0
                                && partnerReward.getSettleState() != Constants.SETTLE_STATE_DONE) {
                            if (isEmpty(partnerReward.getOrderType()) || isEmpty(partnerReward.getOrderSn()) || isEmpty(partnerReward.getRatioAll())) {
                                /**
                                 * 历史数据: 没有记录订单号 或者 没有记录比例 不更新奖励金额
                                 */
                            } else {
                                /**
                                 * 有订单号 和 比例快照 可以更新金额
                                 */
                                OrdersActive ordersActive = ordersActiveService.getOneByOrderSn(partnerReward.getOrderType(), partnerReward.getOrderSn());
                                OrderTaobaoJdCommission order = orderTaobaoJdCommissionService.getOneByOrderSn(partnerReward.getOrderType(), partnerReward.getOrderSn());
                                if (null == ordersActive || null == order) {
                                    /**
                                     * 如果没有找到active 或者 没有找到commission
                                     * 不更新奖励金额
                                     */
                                } else {
                                    /**
                                     * 订单状态只有三种: -3, 1, 3
                                     */
                                    if (order.getStatus() == Constants.ORDER_STATE_FAILED) {
                                        /**
                                         * 订单失效 订单付款 跳过, 处理下一条奖励
                                         */
                                        updateReward2Failed(partnerReward);
                                        continue;
                                    } else if (order.getStatus() == Constants.ORDER_STATE_PAID) {
                                        /**
                                         * 订单还是付款状态 更新奖励时间 下个月再处理
                                         */
                                        long newRechargeTime = DateUtils.getNextMonth1stTimestamp(partnerReward.getRechargeTime().getTime());
                                        partnerReward.setRechargeTime(new Date(newRechargeTime));
                                        partnerRewardService.updateById(partnerReward);
                                        continue;
                                    } else if (order.getStatus() == Constants.ORDER_STATE_SETTLED) {
                                        /**
                                         * 订单结算 更新金额
                                         */
                                        if (ordersActive.getIsForNew() == 1) {
                                            /**
                                             * 新人专享订单 不给合伙人奖励 跳过
                                             */
                                            updateReward2Failed(partnerReward);
                                            continue;
                                        } else {
                                            /**
                                             * 正常订单 更新原始佣金
                                             * all_money_ori: 原始佣金
                                             * cash: all_money_ori 乘以 ratio_all中的每个比例
                                             * commission_account: cash 乘以 commission_rate
                                             */
                                            updateRewardCommission(partnerReward, order.getAllMoney());
                                        }
                                    }
                                }
                            }
                        }
                        /**
                         * 分享赚订单: 按子订单号更新奖励金额
                         */
                        if (partnerReward.getRewardType() == Constants.PARTNER_REWARD_TYPE_SHARE && partnerReward.getSettleState() > 0
                                && partnerReward.getSettleState() != Constants.SETTLE_STATE_DONE) {
                            if (isEmpty(partnerReward.getOrderType()) || isEmpty(partnerReward.getOrderSn()) || isEmpty(partnerReward.getSubOrderSn())
                                    || isEmpty(partnerReward.getRatioAll())) {
                                /**
                                 * 历史数据: 没有记录子订单号 或者 没有记录比例 不更新奖励金额
                                 */
                            } else {
                                /**
                                 * 有订单号 和 比例快照 可以更新金额
                                 */
                                OrderTaobaoJdGoodsCommission subOrder = orderTaobaoJdGoodsCommissionService.getOneByOrderSn(partnerReward.getOrderType(), partnerReward.getOrderSn(), partnerReward.getSubOrderSn());
                                if (null == subOrder) {
                                    /**
                                     * 如果没有找到子订单
                                     * 不更新奖励金额
                                     */
                                } else {
                                    /**
                                     * 订单状态只有三种: -3, 1, 3
                                     */
                                    if (subOrder.getOrderState() == Constants.ORDER_STATE_FAILED) {
                                        /**
                                         * 订单失效 订单付款 跳过, 处理下一条奖励
                                         */
                                        updateReward2Failed(partnerReward);
                                    } else if (subOrder.getOrderState() == Constants.ORDER_STATE_PAID) {
                                        /**
                                         * 订单还是付款状态 更新奖励时间 下个月再处理
                                         */
                                        long newRechargeTime = DateUtils.getNextMonth1stTimestamp(partnerReward.getRechargeTime().getTime());
                                        partnerReward.setRechargeTime(new Date(newRechargeTime));
                                        partnerRewardService.updateById(partnerReward);
                                        continue;
                                    } else if (subOrder.getOrderState() == Constants.ORDER_STATE_SETTLED) {
                                        /**
                                         * 正常订单 更新原始佣金
                                         * all_money_ori: 原始佣金
                                         * cash: all_money_ori 乘以 ratio_all中的每个比例
                                         * commission_account: cash 乘以 commission_rate
                                         */
                                        updateRewardCommission(partnerReward, subOrder.getAllMoney());
                                    }
                                }
                            }
                        }
                    }
                    pageNo++;
                } else {
                    break;
                }
            }
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
            logger.error("errPartnerReward is {}", errPartnerReward);
        }
    }

    private static boolean setIncomeAndTax(PartnerSettle partnerSettle) {
        try {
            BigDecimal base = partnerSettle.getSumRedpack().add(partnerSettle.getSumOthers());
            /**
             * 订单收入全部计入微信 并且不计税
             */
            BigDecimal wxIncome = partnerSettle.getSumOrderCommission();
            BigDecimal wxTax = ZERO;
            BigDecimal bankIncome = ZERO;
            BigDecimal bankTax = ZERO;
//        所有按左开右闭统计
            if (base.compareTo(Constants.TAX_STAGE_FREE_BASE) <= 0) {
                /**
                 * 小于 [0,800]
                 */
            } else if (base.compareTo(Constants.TAX_STAGE_FREE_BASE) == 1 && base.compareTo(Constants.TAX_STAGE_1) <= 0) {
                /**
                 * (800,4000]
                 */
                wxTax = base.subtract(Constants.TAX_STAGE_FREE_BASE).multiply(Constants.TAX_STAGE_1_RATE1).multiply(Constants.TAX_STAGE_1_RATE2);
                wxIncome = wxIncome.add(base).subtract(wxTax);
            } else if (base.compareTo(Constants.TAX_STAGE_1) == 1 && base.compareTo(Constants.TAX_STAGE_WX_MAX) <= 0) {
                /**
                 * (4000-17076.5]
                 */
                wxTax = base.multiply(Constants.TAX_STAGE_WX_MAX_RATE1).multiply(Constants.TAX_STAGE_WX_MAX_RATE2).multiply(Constants.TAX_STAGE_WX_MAX_RATE3);
                wxIncome = wxIncome.add(base).subtract(wxTax);
            } else {
                /**
                 * (17076.5,)
                 *
                 * 微信收入: 17076.5-17076.5*0.8*0.2*0.76
                 */
                wxTax = Constants.TAX_STAGE_WX_MAX.multiply(Constants.TAX_STAGE_WX_MAX_RATE1).multiply(Constants.TAX_STAGE_WX_MAX_RATE2).multiply(Constants.TAX_STAGE_WX_MAX_RATE3);
                wxIncome = wxIncome.add(Constants.TAX_STAGE_WX_MAX).subtract(wxTax);
                bankTax = base.subtract(Constants.TAX_STAGE_WX_MAX).multiply(Constants.BANK_RATE);
                bankIncome = base.subtract(Constants.TAX_STAGE_WX_MAX).subtract(bankTax);
            }
            partnerSettle.setWxIncome(wxIncome);
            partnerSettle.setWxTax(wxTax);
            partnerSettle.setBankIncome(bankIncome);
            partnerSettle.setBankTax(bankTax);
            return true;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            return false;
        }
    }
}
