package com.lanren.huhu.partner.schedule;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.lanren.huhu.partner.domain.AgentReward;
import com.lanren.huhu.partner.domain.PartnerAccount;
import com.lanren.huhu.partner.domain.UserAgent;
import com.lanren.huhu.partner.model.AgentRewardMessage;
import com.lanren.huhu.partner.model.AgentRewardQueueMessage;
import com.lanren.huhu.partner.model.ParentAgent;
import com.lanren.huhu.partner.service.AgentRewardService;
import com.lanren.huhu.partner.service.PartnerAccountService;
import com.lanren.huhu.partner.service.UserAgentService;
import com.lanren.huhu.partner.service.UserService;
import com.lanren.huhu.partner.util.LockUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.ListOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.scheduling.annotation.Async;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import static com.lanren.huhu.partner.constants.Constants.*;

/**
 * @author chen
 * @title: AgentRewardQueueTask
 * @projectName partner
 * @description: 消费 代理商奖励的消息队列, 处理代理商奖励, 写到agent_reward
 * @package com.lanren.huhu.partner.schedule
 * @date 2019-06-29 15:39
 */
@Component
public class AgentRewardQueueTask {
    private static Logger logger = LoggerFactory.getLogger(AgentRewardQueueTask.class);

    @Autowired
    StringRedisTemplate stringRedisTemplate;
    @Autowired
    UserAgentService userAgentService;
    @Autowired
    UserService userService;
    @Autowired
    AgentRewardService agentRewardService;
    @Autowired
    PartnerAccountService partnerAccountService;
    @Autowired
    RedisTemplate redisTemplatePush;
    @Autowired
    DataSourceTransactionManager dataSourceTransactionManager;
    @Autowired
    TransactionDefinition transactionDefinition;

    @Scheduled(fixedDelay = 5000L)
    public void runScheduledTask() {
        logger.info("run AgentRewardQueueTask");
        runConsume();
    }

    private void runConsume() {
        ListOperations<String, String> ops = stringRedisTemplate.opsForList();
        while(null != ops && null != ops.size(AGENT_REWARD_QUEUE_KEY) && ops.size(AGENT_REWARD_QUEUE_KEY) > 0L) {
            String msg = "";
            try {
                /**
                 * 加分布式锁
                 */
                LockUtil.lock(DISTRIBUTE_REDIS_LOCK_KEY);
                try {
                    msg = ops.rightPop(AGENT_REWARD_QUEUE_KEY);
                } catch (Exception e) {
                    logger.error(e.getMessage(), e);
                    continue;
                } finally{
                    LockUtil.unlock(DISTRIBUTE_REDIS_LOCK_KEY);
                }
                logger.info("msg is {}", msg);
                JSONObject json = JSON.parseObject(msg);
                AgentRewardMessage message = json.toJavaObject(AgentRewardMessage.class);
                processReward(message);
            } catch (Exception e) {
                logger.error("process agent reward message failed", msg);
                logger.error(e.getMessage(), e);
            }
        }
    }

    @Async
    public void processReward(AgentRewardMessage message) {
        ListOperations<String, String> ops = stringRedisTemplate.opsForList();
        logger.info("异步处理 agent reward: {}", message);
        int rewardType = message.getRewardType();
        Map<Integer, BigDecimal> rateMap;
        if (!AGENT_RATE_MAP.keySet().contains(rewardType)) {
            logger.info("未知的奖励类型 跳过不处理: {}", message);
            return;
        } else {
            rateMap = AGENT_RATE_MAP.get(rewardType);
        }
        UserAgent userAgent = userAgentService.getOneByAgentId(message.getAgentId());
        if (userAgent == null) {
            logger.info("代理商id: {} 不存在, 跳过不处理", message);
            return;
        }
        if (userAgent.getAgentLevel() != AGENT_LEVEL_4) {
            logger.info("代理商id: {} 不是城市代理, 跳过不处理", message);
            return;
        }
        /**
         * 这里的逻辑有问题, 不能用这个接口找上级, 因为会先找一次直接邀请人,
         * 只能按message.getAgentId()的代理商id, 找user_agent表里的平推关系
         */
        ArrayList<ParentAgent> parentList = (ArrayList<ParentAgent>) userAgentService.getParentAgentListByAgentId(userAgent.getAgentId(), PARENT_COLUMN_NAME_PINGTUI);
        ArrayList<ParentAgent> level4List = new ArrayList<ParentAgent>();
        if (parentList.size() > 0) {
            /**
             * 扫描agentlist中的平推城市代理
             * 最多只需要找3个
             */
            int rewardCnt = 0;
            ArrayList<ParentAgent> rewardList = new ArrayList<ParentAgent>();
            for (ParentAgent agent : parentList) {
                /**
                 * 先扫描平推代理, 写到level4List
                 * 如果不是平推, 奖励次数还不够3, 写到rewardList
                 */
                if (agent.getLevel() == AGENT_LEVEL_4 && rewardCnt < 3) {
                    level4List.add(agent);
                    rewardList.add(agent);
                    rewardCnt++;
                } else if (rewardCnt < 3){
                    rewardList.add(agent);
                    rewardCnt++;
                } else {
                    break;
                }
            }
            /**
             * 如果平推层级小于3, 又没有上级代理, 报错不处理
             */
            if (level4List.size() < 3 && level4List.size() == rewardList.size()) {
                logger.error("代理商关系错误, 只有平推, 没有上级. {}", message);
                return;
            }
            /**
             * 如果上面检查通过, 证明代理商关系没问题, 后面处理只看rewardList的size
             * 决定比例怎么分: 1个人-拿3份; 2个人-第二个拿后2份; 3个人-各1份
             */
            List<BigDecimal> rateList = new ArrayList<BigDecimal>();
            if (rewardList.size() == 1) {
                rateList.add(rateMap.get(0).add(rateMap.get(1)).add(rateMap.get(2)));
            } else if (rewardList.size() == 2) {
                rateList.add(rateMap.get(0));
                rateList.add(rateMap.get(1).add(rateMap.get(2)));
            } else if (rewardList.size() == 3) {
                rateList.add(rateMap.get(0));
                rateList.add(rateMap.get(1));
                rateList.add(rateMap.get(2));
            }

            TransactionStatus transactionStatus = null;
            List<AgentReward> pushList = new ArrayList<AgentReward>();
            try {
                /**
                 * 开启事务, 如果rewardList中的奖励 都插入成功, 才提交事务
                 * 并发送redis消息, 否则, 进行回滚, 并把读入的消息塞回队列
                 */
                transactionStatus = dataSourceTransactionManager.getTransaction(transactionDefinition);
                for (int i=0; i<rewardList.size(); i++) {
                    BigDecimal commissionRate = rateList.get(i);
                    AgentReward reward = doInsert(commissionRate, rewardList.get(i), message);
                    if (reward != null) {
                        pushList.add(reward);
                    }
                }
                /**
                 * 提交事务
                 */
                dataSourceTransactionManager.commit(transactionStatus);
                /**
                 * 数据库写入成功后 发推送
                 */
                for (AgentReward reward : pushList) {
                    doPush(reward);
                }
            } catch (Exception e) {
                logger.error(e.getMessage(), e);
                logger.error("插入代理商分成奖励失败, 奖励 {}", message);
                /**
                 * 回滚事务
                 */
                if (transactionStatus != null) {
                    dataSourceTransactionManager.rollback(transactionStatus);
                }
                /**
                 * 消息处理失败 丢回队列
                 */
                ops.leftPush(AGENT_REWARD_QUEUE_KEY, JSON.toJSONString(message));
                /**
                 * 休眠
                 */
                try {
                    Thread.sleep(5 * 1000L);
                } catch (Exception e1) {
                    logger.error(e.getMessage(), e1);
                }
            }
        }
    }

    private AgentReward doInsert(BigDecimal commissionRate, ParentAgent agent, AgentRewardMessage message) {
        try {
            int rewardType = message.getRewardType();
            int sourceUserLevel = partnerAccountService.getOneByUserId(message.getSourceUserId()).getPartnerLevel();
            AgentReward agentReward = new AgentReward();
            agentReward.setRewardType(message.getRewardType());
            agentReward.setRechargeTime(LocalDateTime.parse(message.getRechargeTime(), DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));
            agentReward.setSourceUserId(message.getSourceUserId());
            agentReward.setReferenceId(Long.parseLong(message.getReferenceId()));
            agentReward.setAgentId(agent.getAgentId());
            agentReward.setSettleState(message.getSettleState());
            agentReward.setOrderType(message.getOrderType());
            agentReward.setOrderSn(message.getOrderSn());
            agentReward.setSubOrderSn(message.getSubOrderSn());
            agentReward.setGoodsId(message.getOrderGoodsId());
            agentReward.setTitle(message.getOrderTitle());
            agentReward.setAgentLevel(agent.getLevel());
            agentReward.setUserId(agent.getUserId());
            agentReward.setRewardRemark(AGENT_REWARD_REMARK.get(message.getRewardType()));
            if (StringUtils.isEmpty(message.getSourceOrder())) {
                agentReward.setSourceOrder(message.getReferenceId());
            } else {
                agentReward.setSourceOrder(message.getSourceOrder());
            }
            agentReward.setSourceUserLevel(sourceUserLevel);
            agentReward.setCreatedAt(System.currentTimeMillis() / 1000L);
            agentReward.setCommissionRate(commissionRate);
            agentReward.setRewardTypeChild(message.getRewardTypeChild() == null ? "" : message.getRewardTypeChild());
            /**
             * agent_reward里:
             * money 面额(message里, 需要上供的代理商自己获得的奖励)
             * commission_account 面额
             * amount 奖励金额
             * commission 奖励金额
             * cash_code 支付金额
             */
            BigDecimal rewardBasement;
            if (rewardType == AGENT_REWARD_TYPE_YEAR_VIP || rewardType == AGENT_REWARD_TYPE_HALF_YEAR_VIP || rewardType == AGENT_REWARD_TYPE_SVIP) {
                rewardBasement = message.getSourceUserPayment();
            } else {
                /**
                 * 订单相关, 还需要写入两个字段:all_money_ori 原始佣金, 继承下来的比例数组
                 */
                rewardBasement = message.getAgentReward();
                agentReward.setAllMoneyOri(message.getOrderCommission());
                agentReward.setRatioAll(message.getOrderRateArray());
            }
            logger.info("rewardBasement is: {}", rewardBasement);
            logger.info("commissionRate is: {}", commissionRate);
            BigDecimal reward = rewardBasement.multiply(commissionRate);
            /**
             * 小于0.000001的奖励 不写入
             */
            if (reward.compareTo(MIN_REWARD) < 0) {
                logger.info("代理商({})平推奖励金额小于0.000001, 忽略不写入. msg:{}", agent.getAgentId(), message);
                return null;
            } else {
                agentReward.setMoney(rewardBasement);
                agentReward.setCommissionAcount(rewardBasement);
                agentReward.setAmount(reward);
                agentReward.setCommission(reward);
                agentReward.setCashCode("0");
                doPush(agentReward);
                agentRewardService.save(agentReward);
                return agentReward;
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            return null;
        }
    }

    private void doPush(AgentReward agentReward) {
        try {
            JSONObject json = new JSONObject();
            PartnerAccount partnerAccount = partnerAccountService.getOneByUserId(agentReward.getUserId());
            AgentRewardQueueMessage message = new AgentRewardQueueMessage();
            message.setUserId(agentReward.getUserId());
            message.setSourceUserId(agentReward.getSourceUserId());
            message.setMoney(agentReward.getCommission().toString());
            message.setPartnerLevel(partnerAccount.getIsSuperPartner() == 1 ? 30 : partnerAccount.getPartnerLevel());
            message.setTime(agentReward.getCreatedAt());
            message.setRewardType(agentReward.getRewardType());
            redisTemplatePush.opsForList().leftPush(AGENT_REWARD_PUSH_KEY, message);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
    }
}
