package com.lanren.huhu.partner.manager;

import com.lanren.huhu.partner.domain.PartnerAccount;
import com.lanren.huhu.partner.domain.UserAgent;
import com.lanren.huhu.partner.model.*;
import com.lanren.huhu.partner.result.Result;
import com.lanren.huhu.partner.service.UserService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.util.ArrayList;

import static com.lanren.huhu.partner.constants.Constants.PARENT_COLUMN_NAME_CENGJI;
import static com.lanren.huhu.partner.constants.Constants.PARENT_COLUMN_NAME_PINGTUI;

/**
 * @author houseme
 */
@Component
public class UserManager {
    private static Logger logger = LoggerFactory.getLogger(UserManager.class);

    @Autowired
    UserService userService;

    public Result<User> getRelationByUserId(int userId) {
        User resutrnUser = userService.getRelationByUserId(userId);
        Result<User> result = new Result<User>();
        result.setData(resutrnUser);
        return result;
    }

    public Result<String> deleteRalationByUserId(int userId) {
        Result<String> result = new Result<String>();
        if (userService.deleteRalationByUserId(userId)) {
            result.setData("删除成功");
        } else {
            result.setCode(500);
        }
        return result;
    }

    /**
     * 根据输入的userId, 完成递归查找
     * @param userId
     */
    @Async
    public Result<String> purgeRelation(Integer userId) {
        Result<String> result = new Result<String>();
        try{
            logger.info("start purgeRelation for userId: {} ...", userId);
            PartnerAccount partnerAccount = userService.getPartnerAccountByUserId(userId);
            if (partnerAccount == null) {
                result.setCode(413);
            }
            int partnerLevel = partnerAccount.getIsSuperPartner() == 1 ? 30 : partnerAccount.getPartnerLevel();
            /**
             * 合伙人按邀请关系
             */
            PartnerTree root = new PartnerTree();
            root.setUserId(partnerAccount.getUserId());
            root.setLevel(partnerLevel);
            root.setParentList(new ArrayList<ParentPartner>());
            recursiveSet(root);
            /**
             * 如果这个用户还是代理商身份, 也要处理他的代理商下级(层级下级 和 平推下级)
             */
            UserAgent agent = userService.getUserAgentByUserId(partnerAccount.getUserId());
            if (null != agent) {
                AgentTree agentRoot = new AgentTree();
                agentRoot.setUserId(partnerAccount.getUserId());
                agentRoot.setAgentId(agent.getAgentId());
                agentRoot.setLevel(partnerLevel);
                agentRoot.setParentList(new ArrayList<ParentAgent>());
                recursiveSet(agentRoot, PARENT_COLUMN_NAME_CENGJI);
                recursiveSet(agentRoot, PARENT_COLUMN_NAME_PINGTUI);
            }
            logger.info("done purgeRelation for userId: {}", userId);
            result.setData("用户刷新成功");
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            result.setCode(500);
        }
        return result;
    }

    /**
     * ****************************************分割线****合伙人***************************************************************
     */

    /**
     * 递归查找, 完成输入的tree的所有下级查找
     * @param tree
     */
    private void recursiveSet(PartnerTree tree) {
        logger.info("do recursiveSet for userId: {}", tree.getUserId());
        if (logger.isDebugEnabled()) {
            logger.info("do recursiveSet for userId: {}", tree.getUserId());
        }
        /**
         * 先找到直接下级
         */
        setChildList(tree);
        ArrayList<PartnerTree> children = tree.getChildList();
        /**
         * 如果没有下级了 那么直接输出结果
         */
        if (children.size() == 0) {
            outputTree(tree);
        } else {
            /**
             * 如果有直接下级, 遍历下级, 对每个下级再执行recursiveSet
             */
            for (PartnerTree child : children) {
                recursiveSet(child);
                /**
                 * 当完成 对每个下级再执行recursiveSet 之后
                 * 判断一下, 每个下级是否有直接下级
                 * 如果有 那么在各自的recursiveSet中 应该没有走到 line:52 的outputTree 方法
                 * 所以要在这里执行一次, 把他们自己输出
                 */
                if (child.getChildList().size() > 0) {
                    outputTree(child);
                }
            }
            /**
             * 最后输出自己
             */
            outputTree(tree);
        }
    }

    /**
     * 完成刷新用户缓存数据
     * @param tree
     */
    private void outputTree(PartnerTree tree) {
        StringBuilder sb = new StringBuilder();
        ArrayList<ParentPartner> list = tree.getParentList();
        for (ParentPartner p : list) {
            sb.append(p.getUserId()).append(",");
        }
        if (logger.isDebugEnabled()) {
            logger.info("begin to clean cache......");
            logger.info("userId: {}, parent is: {}", tree.getUserId(), sb.toString());
        }
        deleteRalationByUserId(tree.getUserId());
        /**
         * 加快接口返回速度 不再为每一个叶子做刷新
         * 只做清楚缓存的操作
         */
//        getRelationByUserId(tree.getUserId());
    }

    /**
     * 无递归操作, 只往下查找一级
     * 把下级的数据, 设置到输入的tree中的childrenList
     * @param tree
     */
    private void setChildList(PartnerTree tree) {
        ArrayList<PartnerAccount> partnerAccountArrayList = (ArrayList<PartnerAccount>) userService.getPartnerAccountChildListByUserId(tree.getUserId());
        for (PartnerAccount partnerAccount : partnerAccountArrayList) {
            Integer childId = partnerAccount.getUserId();
            Integer childLevel = partnerAccount.getIsSuperPartner() == 1 ? 30 : partnerAccount.getPartnerLevel();
            if (!tree.getParentList().contains(childId)) {
                ArrayList<ParentPartner> parentList4Child = tree.cloneParent();
                parentList4Child.add(new ParentPartner(tree.getUserId(), tree.getLevel()));
                PartnerTree oneChild = new PartnerTree();
                oneChild.setUserId(childId);
                oneChild.setLevel(childLevel);
                oneChild.setParentList(parentList4Child);
                tree.getChildList().add(oneChild);
            }
        }
    }

    /**
     * ****************************************分割线****代理商***************************************************************
     */

    /**
     * 递归查找, 完成输入的tree的所有下级查找
     * @param tree
     */
    private void recursiveSet(AgentTree tree, String parentColumnName) {
        logger.info("do recursiveSet for userId: {}", tree.getUserId());
        if (logger.isDebugEnabled()) {
            logger.info("do recursiveSet for userId: {}", tree.getUserId());
        }
        /**
         * 先找到直接下级
         */
        setChildList(tree, parentColumnName);
        ArrayList<AgentTree> children = tree.getChildList();
        /**
         * 如果没有下级了 那么直接输出结果
         */
        if (children.size() == 0) {
            outputTree(tree);
        } else {
            /**
             * 如果有直接下级, 遍历下级, 对每个下级再执行recursiveSet
             */
            for (AgentTree child : children) {
                recursiveSet(child, parentColumnName);
                /**
                 * 当完成 对每个下级再执行recursiveSet 之后
                 * 判断一下, 每个下级是否有直接下级
                 * 如果有 那么在各自的recursiveSet中 应该没有走到 line:52 的outputTree 方法
                 * 所以要在这里执行一次, 把他们自己输出
                 */
                if (child.getChildList().size() > 0) {
                    outputTree(child);
                }
            }
            /**
             * 最后输出自己
             */
            outputTree(tree);
        }
    }

    /**
     * 完成刷新用户缓存数据
     * @param tree
     */
    private void outputTree(AgentTree tree) {
        StringBuilder sb = new StringBuilder();
        ArrayList<ParentAgent> list = tree.getParentList();
        for (ParentAgent p : list) {
            sb.append(p.getUserId()).append(",");
        }
        if (logger.isDebugEnabled()) {
            logger.info("begin to clean cache......");
            logger.info("userId: {}, parent is: {}", tree.getUserId(), sb.toString());
        }
        deleteRalationByUserId(tree.getUserId());
    }

    /**
     * 无递归操作, 只往下查找一级
     * 把下级的数据, 设置到输入的tree中的childrenList
     * @param tree
     */
    private void setChildList(AgentTree tree, String parentColumnName) {
        ArrayList<UserAgent> partnerAccountArrayList = (ArrayList<UserAgent>) userService.getUserAgentChildListByUserId(tree.getUserId(), parentColumnName);
        for (UserAgent userAgent : partnerAccountArrayList) {
            Integer childId = userAgent.getUserId();
            Integer agentId = userAgent.getAgentId();
            Integer childLevel = userAgent.getAgentLevel();
            if (!tree.getParentList().contains(childId)) {
                ArrayList<ParentAgent> parentList4Child = tree.cloneParent();
                parentList4Child.add(new ParentAgent(tree.getUserId(), tree.getAgentId(), tree.getLevel()));
                AgentTree oneChild = new AgentTree();
                oneChild.setUserId(childId);
                oneChild.setAgentId(agentId);
                oneChild.setLevel(childLevel);
                oneChild.setParentList(parentList4Child);
                tree.getChildList().add(oneChild);
            }
        }
    }

    /**
     * 变更 代理商 或 用户的 上级
     * 变更类型: 1改层级关系 2改推荐关系 3改层级和推荐关系 4改C端邀请关系 5全部都改
     */
    @Async
    public Result<String> updateRelation(String from, String to, int type) {
        if (type == 1 || type == 3 || type == 5) {
            int cnt = updateCengjiRelation(from, to);
            logger.info("更新代理商层级关系, 将{}的上级改为{},更新{}行数据", from, to, cnt);
        }
        if (type == 2 || type == 3 || type == 5) {
            int cnt = updatePingtuiRelation(from, to);
            logger.info("更新代理商推荐关系, 将{}的上级改为{},更新{}行数据", from, to, cnt);

        }
        if (type == 4 || type == 5) {
            int cnt = updatePartnerRelation(from, to);
            logger.info("更新合伙人推荐关系, 将{}的上级改为{},更新{}行数据", from, to, cnt);
        }
        PartnerAccount partnerAccount = userService.getPartnerAccountByPhone(from);
        return purgeRelation(partnerAccount.getUserId());
    }

    private int updatePartnerRelation(String from, String to) {
        return userService.updatePartnerRelation(from, to);
    }

    private int updatePingtuiRelation(String from, String to) {
        return userService.updatePingtuiRelation(from, to);
    }

    private int updateCengjiRelation(String from, String to) {
        return userService.updateCengjiRelation(from, to);
    }
}
