package com.lanren.huhu.partner.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.lanren.huhu.partner.dao.PartnerAccountMapper;
import com.lanren.huhu.partner.dao.PartnerInviteRelationMapper;
import com.lanren.huhu.partner.dao.UserAgentMapper;
import com.lanren.huhu.partner.domain.PartnerAccount;
import com.lanren.huhu.partner.domain.PartnerInviteRelation;
import com.lanren.huhu.partner.domain.UserAgent;
import com.lanren.huhu.partner.model.ParentAgent;
import com.lanren.huhu.partner.model.ParentPartner;
import com.lanren.huhu.partner.model.User;
import com.lanren.huhu.partner.service.UserService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import static com.lanren.huhu.partner.constants.Constants.*;

/**
 * @author houseme
 * @date 2019-06-28 18:36
 * @Project partner
 * @Package com.lanren.huhu.partner.service.impl
 * @File: UserServiceImpl
 */
@Service
@CacheConfig(cacheNames = "user:relation:cache")
public class UserServiceImpl implements UserService {
    private static Logger logger = LoggerFactory.getLogger(UserServiceImpl.class);

    @Autowired
    private UserAgentMapper userAgentMapper;
    @Autowired
    private PartnerAccountMapper partnerAccountMapper;
    @Autowired
    private PartnerInviteRelationMapper partnerInviteRelationMapper;


    /**
     * ****************************************分割线****用户邀请关系***************************************************************
     */

    /**
     * 用户上级 合伙人 和 代理 关系
     * @param userId
     * @return
     */
    @Override
    @Cacheable
    public User getRelationByUserId(Integer userId) {
        return getRelationByUserIdRealtime(userId);
    }

    @Override
    public User getRelationByUserIdRealtime(Integer userId) {
        logger.info("UserServiceImpl get user relation from database, userId: {}", userId);
        User user = new User();
        user.setUserId(userId);
        user.setPartnerList((ArrayList<ParentPartner>) getPartnerListByUserId(userId));
        user.setAgentList((ArrayList<ParentAgent>) getAgentListByUserId(userId, PARENT_COLUMN_NAME_CENGJI));
        user.setPingtuiAgentList((ArrayList<ParentAgent>) getAgentListByUserId(userId, PARENT_COLUMN_NAME_PINGTUI));
        return user;
    }

        @Override
    @CacheEvict(key = "#userId")
    public boolean deleteRalationByUserId(Integer userId) {
        logger.info("UserServiceImpl delete user relation cache, userId: {}", userId);
        return true;
    }

    /**
     * ****************************************分割线****合伙人***************************************************************
     */

    /**
     * 合伙人数据
     * @param userId
     * @return
     */
    @Override
    public PartnerAccount getPartnerAccountByUserId(Integer userId) {
        QueryWrapper<PartnerAccount> queryWrapper = new QueryWrapper<PartnerAccount>();
        queryWrapper.eq("user_id", userId);
        return partnerAccountMapper.selectOne(queryWrapper);
    }
    @Override
    public List<PartnerAccount> getPartnerAccountChildListByUserId(Integer userId) {
        return partnerAccountMapper.getChildListByUserId(userId);
    }

    /**
     * 按userId查找上级合伙人的 userId 和 level
     * @param userId
     * @return
     */
    private ParentPartner getParentPartner(Integer userId) {
        PartnerAccount partnerAccount = partnerAccountMapper.getParentByUserId(userId);
        ParentPartner parentPartner = null;
        if (null != partnerAccount) {
            int partnerLevel = partnerAccount.getIsSuperPartner() == 1 ? 30 : partnerAccount.getPartnerLevel();
            parentPartner = new ParentPartner(partnerAccount.getUserId(),partnerLevel);
        }
        return parentPartner;
    }

    /**
     * @description: 查找一个用户的全部合伙人上级
     * @param userId
     * @author chen
     * @return List<ParentPartner>
     */
    private List<ParentPartner> getPartnerListByUserId(Integer userId) {
        if (logger.isInfoEnabled()) {
            logger.info("UserServiceImpl getPartnerListByUserId for user: {} ......", userId);
        }
        ArrayList<ParentPartner> partnerList = new ArrayList<ParentPartner>();
        /**
         * 无限找
         */
        int loopUserId = userId;
        int loopTimes = 0;
        while (true) {
            if (logger.isDebugEnabled()) {
                logger.info("now loop uid is {}", loopUserId);
            }
            ParentPartner parentPartner = getParentPartner(loopUserId);
            if (null == parentPartner) {
                break;
            } else {
                if (logger.isDebugEnabled()) {
                    logger.info("parent is {} ", parentPartner.getUserId());
                }
                if (partnerList.contains(parentPartner) || parentPartner.getUserId() == userId) {
                    logger.info("发现互相邀请的关系 用户({})", parentPartner.getUserId());
                    logger.info("退出循环, 不再继续查找");
                    break;
                }
                partnerList.add(parentPartner);
                loopUserId = parentPartner.getUserId();
            }
            loopTimes++;
            if (loopTimes > 500) {
                partnerList.clear();
                logger.error("用户:{} 邀请关系异常, 已超500层", userId);
                break;
            }
        }
        if (logger.isInfoEnabled()) {
            logger.info("setPartnerList for user: {} end", userId);
        }
        return partnerList;
    }

    @Override
    public int updatePartnerRelation(String from, String to) {
        PartnerAccount fromPartnerAccount = partnerAccountMapper.getPartnerAccountByPhone(from);
        PartnerAccount toPartnerAccount = partnerAccountMapper.getPartnerAccountByPhone(to);
        if (null != fromPartnerAccount && null != toPartnerAccount) {
            QueryWrapper<PartnerInviteRelation> wrapper = new QueryWrapper<PartnerInviteRelation>();
            wrapper.eq("user_id", fromPartnerAccount.getUserId());
            PartnerInviteRelation partnerInviteRelation = partnerInviteRelationMapper.selectOne(wrapper);
            partnerInviteRelation.setInviteUserId(toPartnerAccount.getUserId());
            partnerInviteRelation.setInvitePartnerLevel(toPartnerAccount.getIsSuperPartner() == 1 ? 30 : toPartnerAccount.getPartnerLevel());
            return partnerInviteRelationMapper.updateById(partnerInviteRelation);
        } else {
            return 0;
        }
    }

    @Override
    public PartnerAccount getPartnerAccountByPhone(String phone) {
        return partnerAccountMapper.getPartnerAccountByPhone(phone);
    }


    /**
     * ****************************************分割线****代理商***************************************************************
     */

    /**
     * 代理商数据
     * @param userId
     * @return
     */
    @Override
    public UserAgent getUserAgentByUserId(Integer userId) {
        QueryWrapper<UserAgent> queryWrapper = new QueryWrapper<UserAgent>();
        queryWrapper.eq("user_id", userId).le("agent_level", 4);
        return userAgentMapper.selectOne(queryWrapper);
    }

    @Override
    public List<UserAgent> getUserAgentChildListByUserId(Integer userId, String parentColumnName) {
        return userAgentMapper.getChildListByUserId(userId, parentColumnName);
    }

    @Override
    public int updatePingtuiRelation(String from, String to) {
        UserAgent fromUa = userAgentMapper.getUserAgentByPhone(from);
        UserAgent toUa = userAgentMapper.getUserAgentByPhone(to);
        if (null != fromUa && null != toUa) {
            fromUa.setPresenterId(toUa.getAgentId());
        }
        return userAgentMapper.updateById(fromUa);
    }

    @Override
    public int updateCengjiRelation(String from, String to) {
        UserAgent fromUa = userAgentMapper.getUserAgentByPhone(from);
        UserAgent toUa = userAgentMapper.getUserAgentByPhone(to);
        if (null != fromUa && null != toUa) {
            fromUa.setParentAgentId(toUa.getAgentId());
        }
        return userAgentMapper.updateById(fromUa);
    }

    /**
     * 按agentId 查找上级代理商的 agentId 和 level
     * @param agentId
     * @return
     */
    private ParentAgent getParentAgent(Integer agentId, String parentColumnName) {
        UserAgent userAgent = userAgentMapper.getParentAgent(agentId, parentColumnName);
        ParentAgent parentAgent = null;
        if (null != userAgent) {
            int agentLevel = userAgent.getAgentLevel();
            parentAgent = new ParentAgent(userAgent.getUserId(), userAgent.getAgentId(), agentLevel);
        }
        return parentAgent;
    }

    /**
     * @description: 查找一个用户的全部代理商上级
     * 注意 !!!!!!!!  这个方法是按输入用户的邀请关系链上查找 第一个有agent_level<=4的代理商身份的邀请人, 然后再走他的代理关系
     * 注意 !!!!!!!!  如果需要获取一个代理商的上级代理 需要从UserAgentService里的方法获取
     * @param userId
     * @return List<ParentAgent>
     */
    @Override
    public List<ParentAgent> getAgentListByUserId(Integer userId, String parentColumnName) {
        if (logger.isInfoEnabled()) {
            logger.info("getAgentListByUserId {} for user: {} ......", parentColumnName, userId);
        }
        ArrayList<ParentAgent> agentList = new ArrayList<ParentAgent>();
        HashSet<Integer> existingUserIdSet = new HashSet<Integer>();
        /**
         * 无限找
         * 先无限找上级合伙人, 直到找到第一个是代理商的合伙人
         */

        /**
         * 要改成从自己开始找 所以添加判断 看自己是不是
         */
        UserAgent selfAgent = getUserAgentByUserId(userId);
        /**
         *   代理商关系包含自己的条件:
         *   1. 层级关系 都包含自己
         *   2. 推荐关系 只有当自己是总代时才包含自己
         */

        /**
         * 1.4.5更新: 平推奖励都不包含自己了
         */
        int loopUserId = userId;
        int loopTimes = 0;
        UserAgent firstAgent = null;
        /**
         * 自己不是代理商的话 按C端邀请关系找到第一个代理商
         * 自己是代理商的话 按B端找, 如果是找平推, 最后都找完再剔除自己
         */
        if (selfAgent == null) {
            /**
             * 自己不是代理商时 按C端邀请关系查找代理商
             */
            while (true) {
                if (logger.isDebugEnabled()) {
                    logger.info("now loop uid is {}", loopUserId);
                }
                existingUserIdSet.add(loopUserId);
                ParentPartner parentPartner = getParentPartner(loopUserId);
                if (parentPartner != null) {
                    UserAgent agent = getUserAgentByUserId(parentPartner.getUserId());
                    if (agent != null) {
                        firstAgent = agent;
                        break;
                    } else {
                        /**
                         * 这个人不是代理商 继续往上找
                         */
                        loopUserId = parentPartner.getUserId();
                        if (existingUserIdSet.contains(loopUserId)) {
                            logger.info("发现互为上级的用户邀请关系 用户ID({})", loopUserId);
                            logger.info("退出循环, 不再继续查找");
                            break;
                        }
                    }
                } else {
                    /**
                     * 没有上级合伙人 打印结束信息 跳出循环
                     */
                    if (logger.isDebugEnabled()) {
                        logger.info("loop end with userId: {}", loopUserId);
                    }
                    break;
                }
                loopTimes++;
                if (loopTimes > 500) {
                    agentList.clear();
                    logger.error("用户ID:{} 合伙人关系异常, 已超500层", userId);
                    break;
                }
            }
        } else {
            firstAgent = selfAgent;
        }

        /**
         * 找到了第一个是代理商的合伙人
         */
        if (null != firstAgent) {
            /**
             * 然后开始 无限找 这个人的上级代理商
             */
            if (logger.isDebugEnabled()) {
                logger.info("find first agent({}) for user: {}", firstAgent.getAgentId(), userId);
            }
            /**
             * 添加代理到list时 都需要加个判断 如果找到总代 就不再找了
             */
            agentList.add(new ParentAgent(firstAgent.getUserId(), firstAgent.getAgentId(), firstAgent.getAgentLevel()));
            if (firstAgent.getAgentLevel() != AGENT_LEVEL_1) {
                loopTimes = 0;
                if (logger.isDebugEnabled()) {
                    logger.info("loop find parent agent start ......");
                }
                int loogAgentId = firstAgent.getAgentId();
                loopTimes = 0;
                while (true) {
                    ParentAgent loopUserAgent = getParentAgent(loogAgentId, parentColumnName);
                    if (null == loopUserAgent) {
                        break;
                    } else {
                        if (logger.isDebugEnabled()) {
                            logger.info("parent agent is ", loopUserAgent.getAgentId());
                        }
                        if (agentList.contains(loopUserAgent) || loopUserAgent.getAgentId() == firstAgent.getAgentId()) {
                            logger.info("发现互为上级的代理关系 代理商({})", loopUserAgent.getAgentId());
                            logger.info("退出循环, 不再继续查找");
                            break;
                        }
                        /**
                         * 添加代理到list时 都需要加个判断 如果找到总代 就不再找了
                         */
                        agentList.add(new ParentAgent(loopUserAgent.getUserId(), loopUserAgent.getAgentId(), loopUserAgent.getLevel()));
                        if (loopUserAgent.getLevel() == AGENT_LEVEL_1) {
                            break;
                        }
                        loogAgentId = loopUserAgent.getAgentId();
                    }
                    loopTimes++;
                    if (loopTimes > 500) {
                        agentList.clear();
                        logger.error("用户ID:{} 合伙人关系异常, 已超500层", userId);
                        break;
                    }
                }
                /**
                 * 找完 "第一个是代理商的合伙人" 的 全部上级代理后 退出while循环
                 */
                if (logger.isDebugEnabled()) {
                    logger.info("loop find parent agent end");
                }
            } else {
                if (logger.isInfoEnabled()) {
                    logger.info("已找到总代 不再往下找, 总代id: {}", firstAgent.getAgentId());
                }
            }
            /**
             * 如果是平推奖励 过滤掉自己
             */
            if (parentColumnName.equals(PARENT_COLUMN_NAME_PINGTUI)
                    && selfAgent != null
                    && agentList.size() > 0) {
                agentList.remove(0);
            }
        }

        if (logger.isInfoEnabled()) {
            logger.info("setAgentList {} for user: {} end", parentColumnName, userId);
        }
        return agentList;
    }
}