package com.lanren.huhu.partner.util;

import io.netty.util.internal.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.Timestamp;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 日期工具类
 * <p>
 * 默认使用 "yyyy-MM-dd HH:mm:ss" 格式化日期
 *
 * @author houseme
 * @version1.0
 */

public final class DateUtils {
    private static final Logger logger = LoggerFactory.getLogger(DateUtils.class);
    /**
     * 英文简写（默认）如：2010-12-01
     */

    public static String FORMAT_SHORT = "yyyy-MM-dd";

    /**
     * 英文最简写（默认）如：20101201
     */

    public static String FORMAT_MORE_SHORT = "yyyyMMdd";

    /**
     * 英文全称 如：2010-12-01 23:15:06
     */

    public static String FORMAT_LONG = "yyyy-MM-dd HH:mm:ss";

    /**
     * 英文全称 如：2010-12-01 23:15
     */

    public static String FORMAT_CAR_DATE = "yyyy-MM-dd HH:mm";

    /**
     * 精确到毫秒的完整时间 如：yyyy-MM-dd HH:mm:ss.S
     */

    public static String FORMAT_FULL = "yyyy-MM-dd HH:mm:ss.S";

    /**
     * 中文简写 如：2010年12月01日
     */

    public static String FORMAT_SHORT_CN = "yyyy年MM月dd";

    /**
     * 中文全称 如：2010年12月01日 23时15分06秒
     */

    public static String FORMAT_LONG_CN = "yyyy年MM月dd日  HH时mm分ss秒";

    /**
     * 精确到毫秒的完整中文时间 yyyy年MM月dd日 HH时mm分ss秒SSS毫秒
     */

    public static String FORMAT_FULL_CN = "yyyy年MM月dd日  HH时mm分ss秒SSS毫秒";

    /**
     * 精确到秒的完整中文最简时间 yyyyMMddHHmmss
     */
    public static String FORMAT_FULL_SIMPLE = "yyyyMMddHHmmss";

    /**
     * 精确到秒的完整中文最简时间 HHmmss 如：100101
     */
    public static String FORMAT_TIME_FULL = "HHmmss";

    /**
     * 精确到毫秒的完整中文最简时间 如：20150304230516221
     */
    public static String FORMAT_TIME_FULL_MILLI = "yyyyMMddHHmmssSSS";

    /**
     * 日期格式正则表达式
     */
    public static String DATA_PATTERN = "^((\\d{2}(([02468][048])|([13579][26]))[\\-\\/\\s]?((((0?[13578])|(1[02]))[\\-\\/\\s]?((0?[1-9])|([1-2][0-9])|(3[01])))|(((0?[469])|(11))[\\-\\/\\s]?((0?[1-9])|([1-2][0-9])|(30)))|(0?2[\\-\\/\\s]?((0?[1-9])|([1-2][0-9])))))|(\\d{2}(([02468][1235679])|([13579][01345789]))[\\-\\/\\s]?((((0?[13578])|(1[02]))[\\-\\/\\s]?((0?[1-9])|([1-2][0-9])|(3[01])))|(((0?[469])|(11))[\\-\\/\\s]?((0?[1-9])|([1-2][0-9])|(30)))|(0?2[\\-\\/\\s]?((0?[1-9])|(1[0-9])|(2[0-8]))))))(\\s(((0?[0-9])|([1-2][0-3]))\\:([0-5]?[0-9])((\\s)|(\\:([0-5]?[0-9])))))?$";

    /**
     * 短格式日期变长格式日期附加字符串，用于补充开始查询时间
     */
    public static String ADDTIONAL_STR_BEGAIN = "2000-01-01 00:00:00";

    /**
     * 短格式日期变长格式日期附加字符串，用于补充结束查询时间
     */
    public static String ADDTIONAL_STR_END = "2000-01-01 23:59:59";

    /**
     * 获得默认的 date pattern
     */

    public static String getDatePattern() {
        return FORMAT_LONG;

    }

    /**
     * 获得默认的 date pattern
     */

    public static String getCarDatePattern() {
        return FORMAT_CAR_DATE;

    }

    /**
     * <p>
     * 功能描述:[方法功能中文描述]
     * </p>
     *
     * @return
     */
    public static Date getDate() {
        return parse(getNow());
    }

    public static Date getDate(String format) {
        return parse(getNow(), format);
    }

    /**
     * 根据预设格式返回当前日期
     *
     * @return
     */
    public static String getNow() {
        return format(new Date());

    }

    /**
     * 根据用户格式返回当前日期
     *
     * @param format
     * @return
     */
    public static String getNow(String format) {
        return format(new Date(), format);

    }

    /**
     * 使用预设格式格式化日期
     *
     * @param date
     * @return
     */

    public static String format(Date date) {

        return format(date, getDatePattern());

    }

    /**
     * 使用预设格式格式化日期
     *
     * @param date
     * @return
     */

    public static String carDateFormat(Date date) {

        return format(date, getCarDatePattern());

    }

    /**
     * 使用用户格式格式化日期
     *
     * @param date    日期
     * @param pattern 日期格式
     * @return
     */

    public static String format(Date date, String pattern) {
        String returnValue = "";
        if (date != null) {
            SimpleDateFormat df = new SimpleDateFormat(pattern);
            returnValue = df.format(date);
        }
        return (returnValue);
    }

    /**
     * 使用预设格式提取字符串日期
     *
     * @param strDate 日期字符串
     * @return
     */
    public static Date parse(String strDate) {
        return parse(strDate, getDatePattern());
    }

    /**
     * 使用yyyy-MM-dd格式提取字符串日期
     *
     * @param strDate 日期字符串
     * @return
     */
    public static Date parseYMD(Date strDate) {

        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");

        return parse(dateFormat.format(strDate), "yyyy-MM-dd");
    }

    /**
     * 使用预设格式提取字符串日期
     *
     * @param strDate 日期字符串
     * @return
     */
    public static Date parseDate(String strDate) {
        return parse(strDate, getCarDatePattern());
    }

    /**
     * 使用用户格式提取字符串日期
     *
     * @param strDate 日期字符串
     * @param pattern 日期格式
     * @return
     */
    public static Date parse(String strDate, String pattern) {
        if (StringUtil.isNullOrEmpty(strDate)) {
            return null;
        }
        SimpleDateFormat df = new SimpleDateFormat(pattern);
        try {
            return df.parse(strDate);
        } catch (ParseException e) {
            return null;
        }
    }

    /**
     * 在日期上增加数个整月
     *
     * @param date 日期
     * @param n    要增加的月数
     * @return
     */
    public static Date addMonth(Date date, int n) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.MONTH, n);
        return cal.getTime();
    }

    /**
     * 在日期上增加天数
     *
     * @param date 日期
     * @param n    要增加的天数
     * @return
     */
    public static Date addDay(Date date, int n) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.DATE, n);
        return cal.getTime();
    }

    public static String addDay(Date date, int i, String format) {
        Date result = addDay(date, i);
        return format(result, format);
    }

    /**
     * 在当前日期上添加小时数
     *
     * @param date
     * @param hour
     * @return
     */
    public static Date addHour(Date date, Double hour) {
        Long time = date.getTime();
        Double times = time + hour * 60 * 60 * 1000;
        return new Date(Math.round(times));
    }

    public static Date subtractionHour(Date date, Double hour) {
        Long time = date.getTime();
        Double times = time - hour * 60 * 60 * 1000;
        return new Date(Math.round(times));
    }

    /**
     * 在当前日期上添加分钟数
     *
     * @param date
     * @param mins
     * @return
     */
    public static Date addMinute(Date date, Double mins) {
        Long time = date.getTime();
        Double times = time + mins * 60 * 1000;
        return new Date(Math.round(times));
    }

    /**
     * 在当前日期上添加分钟数
     *
     * @param date
     * @param mins
     * @return
     */
    public static Date addMinute(Date date, int mins) {
        Long time = date.getTime();
        Double times = time + Double.valueOf(mins) * 60 * 1000;
        return new Date(Math.round(times));
    }

    /**
     * 获取时间戳
     */
    public static String getTimeString() {
        SimpleDateFormat df = new SimpleDateFormat(FORMAT_FULL);
        Calendar calendar = Calendar.getInstance();
        return df.format(calendar.getTime());
    }

    /**
     * 获取日期年份
     *
     * @param date 日期
     * @return
     */
    public static String getYear(Date date) {
        return format(date).substring(0, 4);
    }
    /**
     * 获取日期年份
     *
     * @param date 日期
     * @return
     */
    public static String getMonth(Date date) {
        return format(date).substring(5, 7);
    }
    /**
     * 获取日期年份
     *
     * @param date 日期
     * @return
     */
    public static String getDay(Date date) {
        return format(date).substring(8, 10);
    }

    /**
     * 获取yyyy-MM-dd格式日期
     *
     * @param date
     * @return
     */
    public static String getShortDay(Date date) {
        return format(date, FORMAT_SHORT);
    }

    /**
     * 按默认格式的字符串距离今天的天数
     *
     * @param date 日期字符串
     * @return
     */
    public static int countDays(String date) {
        long t = Calendar.getInstance().getTime().getTime();
        Calendar c = Calendar.getInstance();
        c.setTime(parse(date));
        long t1 = c.getTime().getTime();
        return (int) ((t / 1000 - t1 / 1000) / 3600 / 24);
    }

    /**
     * 按用户格式字符串距离今天的天数
     *
     * @param date   日期字符串
     * @param format 日期格式
     * @return
     */
    public static int countDays(String date, String format) {
        long t = Calendar.getInstance().getTime().getTime();
        Calendar c = Calendar.getInstance();
        c.setTime(parse(date, format));
        long t1 = c.getTime().getTime();
        return (int) ((t / 1000 - t1 / 1000) / 3600 / 24);
    }

    /**
     * 计算两个日期的间隔天数
     *
     * @return
     */
    public static int countDays(Date start, Date end) {
        long t1 = start.getTime();
        long t2 = end.getTime();
        return (int) ((t2 / 1000 - t1 / 1000) / 3600 / 24);
    }

    /**
     * 计算两个日期格式为yyyy-MM-dd HH:mm:ss的月份数
     *
     * @param start
     * @param end
     * @return
     */
    public static int countMonth(String start, String end) {
        int month = 0;
        int year = 0;
        Calendar cal1 = Calendar.getInstance();
        Calendar cal2 = Calendar.getInstance();
        cal1.setTime(parse(start));
        cal2.setTime(parse(end));
        if (cal1.after(cal2)) {
            Calendar temp = cal1;
            cal1 = cal2;
            cal2 = temp;
        }
        year = Math.abs(cal2.get(Calendar.YEAR) - cal1.get(Calendar.YEAR));
        month = cal2.get(Calendar.MONTH) - cal1.get(Calendar.MONTH);
        return year * 12 + month;
    }

    /**
     * 计算两个格式为yyyy-MM-dd日期的月份数
     *
     * @param start
     * @param end
     * @return
     */
    public static int countMonth(Date start, Date end) {
        int month = 0;
        int year = 0;
        Calendar cal1 = Calendar.getInstance();
        Calendar cal2 = Calendar.getInstance();
        cal1.setTime(start);
        cal2.setTime(end);
        if (cal1.after(cal2)) {
            Calendar temp = cal1;
            cal1 = cal2;
            cal2 = temp;
        }
        year = Math.abs(cal2.get(Calendar.YEAR) - cal1.get(Calendar.YEAR));
        month = cal2.get(Calendar.MONTH) - cal1.get(Calendar.MONTH);
        return year * 12 + month;
    }

    /**
     * 忽略年份和月份，只计算天数
     *
     * @param start
     * @param end
     * @return
     */
    public static int countDaysIngoreMonthAndYear(String start, String end) {
        int result = 0;
        Calendar cal1 = Calendar.getInstance();
        Calendar cal2 = Calendar.getInstance();
        cal1.setTime(DateUtils.parse(start));
        cal2.setTime(DateUtils.parse(end));
        result = cal2.get(Calendar.DATE) - cal1.get(Calendar.DATE);
        return result;
    }

    /**
     * 计算两个日期间隔天数
     *
     * @param start  开始日期
     * @param end    结束日期
     * @param isCeil 是否向上取整
     */
    public static int countDays(Date start, Date end, boolean isCeil) {
        if (!isCeil) {// 不满一天则不计算在内
            return countDays(start, end);
        }
        int mins = countMins(start, end);
        double hours = mins / 60.0 / 24;// 算出double类型的小时数
        int intHours = (int) hours;
        if (hours > intHours) {
            intHours++;
        }
        return intHours;
    }

    /**
     * 计算间隔小时数 -带半小时(不足半小时按照半小时 超过半小时按照一个小时算)
     *
     * @return
     */
    public static double countHour(Date start, Date end) {
        long t1 = start.getTime();
        long t2 = end.getTime();
        int hour = (int) ((t2 / 1000 - t1 / 1000) / 3600);
        double halfHour = 0;
        if ((countMins(start, end) - hour * 60) > 30) {
            halfHour = 1;
        } else if ((countMins(start, end) - hour * 60) <= 30 && (countMins(start, end) - hour * 60) > 0) {
            halfHour = 0.5;
        }
        return hour + halfHour;
    }

    /**
     * 计算两个日期的间隔分钟数
     *
     * @return
     */
    public static int countMins(Date start, Date end) {
        start = parse(format(start, FORMAT_CAR_DATE) + ":00", FORMAT_LONG);
        end = parse(format(end, FORMAT_CAR_DATE) + ":00", FORMAT_LONG);
        long t1 = start.getTime();
        long t2 = end.getTime();
        return (int) ((t2 - t1) / 1000 / 60);
    }

    /**
     * 得到昨天0点
     */
    public static Date getYester0Date() {
        return parse(format(addDay(new Date(), -1), FORMAT_SHORT), FORMAT_SHORT);
    }

    /**
     * 得到昨天24点
     */
    public static Date getYester24Date() {
        return parse(format(new Date(), FORMAT_SHORT), FORMAT_SHORT);
    }

    /**
     * 判断是否是长短租
     *
     * @param startDate
     * @param enDate
     * @return
     */
    public static boolean isMoreThenMonth(Date startDate, Date enDate) {
        Calendar c = Calendar.getInstance();
        c.setTime(startDate);
        c.add(Calendar.MONTH, 1);
        if (enDate.getTime() - c.getTimeInMillis() >= 0) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * 计算两个日期之间的时长
     *
     * @param startDate
     * @param endDate
     * @return
     */
    public static String countDateLength(Date endDate, Date startDate) {

        long diff = endDate.getTime() - startDate.getTime();
        long days = diff / (1000 * 60 * 60 * 24);
        long hours = (diff - days * (1000 * 60 * 60 * 24)) / (1000 * 60 * 60);
        long minutes = (diff - days * (1000 * 60 * 60 * 24) - hours * (1000 * 60 * 60)) / (1000 * 60);
        String hireTimeStr = days + "天" + hours + "小时" + minutes + "分";
        return hireTimeStr;
    }

    public static String countDateLength(String endDate, String startDate, String pattern) {
        return countDateLength(parse(endDate, pattern), parse(startDate, pattern));
    }

    /**
     * 根据分钟得到小时数，不足一小时按一小时
     *
     * @param minute 分钟数
     * @param isCeil 是否自动填满
     * @return
     */
    public static int countHours(int minute, boolean isCeil) {
        int hours = 0;
        if (minute == 0) {
            return hours;
        }
        if (isCeil) {
            if (minute % 60 == 0) {
                hours = minute / 60;
            } else {
                hours = minute / 60 + 1;
            }
        } else {
            hours = minute / 60;
        }
        return hours;
    }

    /**
     * 根据分钟得到天数，不足一天按一天算
     *
     * @param minute 分钟数
     * @return
     */
    public static int countDays(int minute) {
        int days = 0;
        if (minute == 0) {
            return days;
        }
        if (minute % 1440 == 0) {
            days = minute / 1440;
        } else {
            days = minute / 1440 + 1;
        }
        return days;
    }

    /**
     * 设置时间的秒为0
     *
     * @param date
     * @return
     */
    public static Date setSecondZero(Date date) {
        String time = format(date, "yyyyMMddHHmm");
        date = parse(time, "yyyyMMddHHmm");
        return date;
    }

    /**
     * 验证字符串是否为日期格式
     *
     * @param strDate
     * @return
     */
    public static boolean isDateFormat(String strDate) {
        if (strDate == null) {
            return Boolean.FALSE;
        }
        Pattern pattern = Pattern.compile(DATA_PATTERN);
        Matcher matcher = pattern.matcher(strDate);
        return matcher.matches();
    }

    /**
     * 验证字符串是否为日期格式
     *
     * @param dateTime
     * @return
     */
    public static boolean isDateTimeFormat(String dateTime, String patternStr) {
        if (StringUtil.isNullOrEmpty(dateTime) || StringUtil.isNullOrEmpty(patternStr)) {
            return Boolean.FALSE;
        }

        SimpleDateFormat dfs = new SimpleDateFormat(patternStr);
        try {
            // 设置lenient为false.
            // 否则SimpleDateFormat会比较宽松地验证日期，比如2007-02-29会被接受，并转换成2007-03-01
            dfs.setLenient(false);
            dfs.parse(dateTime);
            return Boolean.TRUE;
        } catch (Exception e) {
            // 如果throw java.text.ParseException或者NullPointerException，就说明格式不对
            return Boolean.FALSE;
        }
    }

    /**
     * 获取生成长格式日期字符串，用于日期型数据的查询条件
     *
     * @param strDate       日期格式字符串，支持yyyy-MM-dd、....、yyyy-MM-dd HH:mm:ss之时的格式
     * @param additionalStr 用于补充strDate以满足 yyyy-MM-dd HH:mm:ss 格式<br/> 详见ADDTIONAL_STR_BEGAIN，ADDTIONAL_STR_END
     * @return
     */
    public static String getLongFormatDateStr(String strDate, String additionalStr) {
        // 验证输入参数
        if (strDate == null || strDate.trim().length() < 10 || additionalStr == null
                || additionalStr.trim().length() != 19) {
            return null;
        }

        return strDate + additionalStr.substring(strDate.length());
    }

    /**
     * 根据一个date时间 获取 day天前的日期 如：2016-6-17 HH:mm:ss 前30天 2016-5-18 HH:mm:ss
     *
     * @param d
     * @param day
     * @return
     */
    public static Date getDateBefore(Date d, int day) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(d);
        calendar.set(Calendar.DATE, calendar.get(Calendar.DATE) - day);
        return calendar.getTime();
    }

    /**
     * 根据一个date时间 获取 day天后的日期 如：2016-5-18 HH:mm:ss 后30天 2016-6-17 HH:mm:ss
     *
     * @param d
     * @param day
     * @return
     */
    public static Date getDateAfter(Date d, int day) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(d);
        calendar.set(Calendar.DATE, calendar.get(Calendar.DATE) + day);
        return calendar.getTime();
    }

    /**
     * 计算两个日期时间相差几天,几小时
     *
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param format    格式
     * @param tag       比较返回的格式，D表示为天；H表示为小时，M表示为分钟，S表示为秒
     * @return
     */
    public static String dateDiff(String startTime, String endTime, String format, String tag) {
        // 按照传入的格式生成一个simpledateformate对象
        SimpleDateFormat sd = new SimpleDateFormat(format);
        long nd = 1000 * 24 * 60 * 60;// 一天的毫秒数
        long nh = 1000 * 60 * 60;// 一小时的毫秒数
        long nm = 1000 * 60;// 一分钟的毫秒数
        // 获得两个时间的毫秒时间差异
        long diff = 0;
        try {
            diff = sd.parse(endTime).getTime() - sd.parse(startTime).getTime();
        } catch (ParseException e) {
            throw new RuntimeException();
        }
        long day = diff / nd;// 计算差多少天
        long hour = diff % nd / nh;// 计算差多少小时
        long min = diff % nd % nh / nm;// 计算差多少分钟
        StringBuilder diffs = new StringBuilder();

        if (day != 0) {
            diffs.append(day + "天");
        }
        if (hour != 0) {
            diffs.append(hour + "小时");
        }
        if (min != 0) {
            diffs.append(min + "分钟");
        }
        return diffs.toString();
    }

    /**
     * 两个时间点的时间差(秒)（目前微信使用）
     *
     * @return
     * @throws ParseException
     * @param: mintime  最小时间
     * @param: maxtime  最大时间
     */
    public static int diffSecond(String mintime, String maxtime) throws ParseException {
        SimpleDateFormat dfs = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        Date begin = dfs.parse(mintime);
        Date end = dfs.parse(maxtime);
        long between = (end.getTime() - begin.getTime()) / 1000;// 除以1000是为了转换成秒
        return new Long(between).intValue();
    }

    /**
     * 时格式时间真是时间
     */
    public static Date formatWorldTimeToGMTTime(String worldTime) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
        sdf.setTimeZone(TimeZone.getTimeZone("GMT"));
        try {
            return sdf.parse(worldTime);
        } catch (ParseException e) {
            e.printStackTrace();
            return null;
        }
    }


    /**
     * 获取指定天的0:0:0
     *
     * @param day
     * @return
     */
    public static Date getYesterFastDate(Date day) {
        Calendar cl = Calendar.getInstance();
        cl.setTime(day);
        cl.set(Calendar.HOUR_OF_DAY, 0);
        cl.set(Calendar.MINUTE, 0);
        cl.set(Calendar.SECOND, 0);
        cl.set(Calendar.MILLISECOND, 0);
        return cl.getTime();
    }

    /**
     * 获取指定天数的23:59:59
     *
     * @param day
     * @return
     */
    public static Date getYesterLastDate(Date day) {
        Calendar cl = Calendar.getInstance();
        cl.setTime(day);
        cl.set(Calendar.HOUR_OF_DAY, 23);
        cl.set(Calendar.MINUTE, 59);
        cl.set(Calendar.SECOND, 59);
        cl.set(Calendar.MILLISECOND, 999);
        return cl.getTime();
    }

    // 默认时间格式
    private static final String DATE_TIME_FORMAT = "yyyy-MM-dd HH:mm:ss";
    //private static final String TIME_FORMAT = "HH:mm:ss";

    /**
     * 计算两个日期时间相差分钟
     * @param dateTime1: yyyy-MM-dd HH:mm:ss
     * @param dateTime2: yyyy-MM-dd HH:mm:ss
     * @return
     */
    public static long minuteBetween(String dateTime1, String dateTime2){
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        long diff = 0;
        Date d1,d2;
        try {
            d1 = df.parse(dateTime1);
            d2 = df.parse(dateTime2);
            diff = (d1.getTime() - d2.getTime())/1000/60;
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return diff;
    }

    /**
     * 计算两个日期时间相差分钟
     * @param dateTime1:
     * @param dateTime2: yyyy-MM-dd HH:mm:ss
     * @return
     */
    public static long minuteBetween(Date dateTime1, Date dateTime2){
        long diff = 0;
        try {
            diff = (dateTime1.getTime() - dateTime2.getTime())/1000/60;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return diff;
    }

    /**
     * 获取当前时间,默认 24小时制 "yyyy-MM-dd HH:mm:ss"
     */
    public static String getCurrentTimeMillis() {
        Date nowTime = new Date(System.currentTimeMillis());
        SimpleDateFormat sdFormatter = new SimpleDateFormat(DATE_TIME_FORMAT);
        String retStrFormatNowDate = sdFormatter.format(nowTime);
        return retStrFormatNowDate;
    }

    /**
     * 获取当前日期 yyyy-MM-dd
     * @return
     */
    public static String getToday(){
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        String today = sdf.format(new Date());
        return today;
    }

    /**
     *  获取明天日期 yyyyMMdd  下一天
     * @return
     */
    public static String getTomorrow(){
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        calendar.add(Calendar.DATE, 1);
        String tomorrow = sdf.format(calendar.getTime());
        return tomorrow;
    }

    /**
     * 将字符串yyyy-MM-dd转换为时间戳
     * @param y_m_d
     * @param hour
     * @return
     */
    public static long getZeroTimeByDateStr(String y_m_d, int hour) {
        SimpleDateFormat sdf =new SimpleDateFormat("yyyy-MM-dd");
        Date date;
        Calendar now = Calendar.getInstance();
        try {
            date = sdf.parse(y_m_d);
            now.setTime(date);
            now.set(Calendar.HOUR_OF_DAY, hour);
            now.set(Calendar.MINUTE, 0);
            now.set(Calendar.SECOND, 0);
            now.set(Calendar.MILLISECOND, 0);
        } catch (ParseException e) {
            e.printStackTrace();
        }

        return now.getTimeInMillis();
    }

    /**
     * 将字符串yyyy-MM-dd转换为Date
     * @param y_m_d
     * @return
     */
    public static Date getDateByDateStr(String y_m_d) {
        SimpleDateFormat sdf =new SimpleDateFormat("yyyy-MM-dd");
        Date date=null;
        try {
            date = sdf.parse(y_m_d);
        } catch (ParseException e) {
            e.printStackTrace();
        }

        return date;
    }



    public static String date2StringDate(java.sql.Date date) {
        if (date == null) {
            return "";
        }

        DateFormat df = new SimpleDateFormat("yyyy-MM-dd");
        return df.format(date);
    }

    public static String date2StringDateNoLine(java.sql.Date date) {
        if (date == null) {
            return "";
        }

        DateFormat df = new SimpleDateFormat("yyyyMMdd");
        return df.format(date);
    }

    public static String date2StringDatetime(java.sql.Date date) {
        if (date == null) {
            return "";
        }

        DateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        return df.format(date);
    }

    public static String date2StringTime(java.sql.Date date) {
        if (date == null) {
            return "";
        }

        DateFormat df = new SimpleDateFormat("HH:mm:ss");
        return df.format(date);
    }

    public static String timestamp2StringDate(Timestamp t) {
        if (t == null) {
            return "";
        }

        DateFormat df = new SimpleDateFormat("yyyy-MM-dd");
        return df.format(t);
    }

    public static String timestamp2StringTime(Timestamp t) {
        if (t == null) {
            return "";
        }

        DateFormat df = new SimpleDateFormat("HH:mm:ss");
        return df.format(t);
    }

    public static String timestamp2StringDatetime(Timestamp t) {
        if (t == null) {
            return "";
        }

        DateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        return df.format(t);
    }

    public static String timestamp2StringDateHourMin(Timestamp t) {
        if (t == null) {
            return "";
        }
        DateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm");
        return df.format(t);
    }

    public static java.sql.Date string2Date(String str) throws ParseException {
        DateFormat df = new SimpleDateFormat("yyyy-MM-dd");
        return new java.sql.Date(df.parse(str).getTime());
    }

    public static java.sql.Date string2DateNoLine(String str) throws ParseException {
        DateFormat df = new SimpleDateFormat("yyyyMMdd");
        return new java.sql.Date(df.parse(str).getTime());
    }

    public static java.sql.Date string2Datetime(String str) {
        try {
            DateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            return new java.sql.Date(df.parse(str).getTime());
        } catch (ParseException e) {
            logger.error("日期格式化错误！", e);
            throw new RuntimeException("日期格式化错误！");
        }
    }

    public static java.sql.Date string2Time(String str) throws ParseException {
        DateFormat df = new SimpleDateFormat("HH:mm:ss");
        return new java.sql.Date(df.parse(str).getTime());
    }

    public static Timestamp string2Timestamp(String str) throws ParseException {
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        java.util.Date date = df.parse(str);
        String time = df.format(date);
        return Timestamp.valueOf(time);
    }

    public static String date2StringMonthDate(java.sql.Date date) {
        if (date == null) {
            return "";
        }

        DateFormat df = new SimpleDateFormat("yyyyMM");
        return df.format(date);
    }

    // T+1 非工作日
    public static java.sql.Date periodStartTime(Integer period) {
        Calendar time = Calendar.getInstance();
        int day = time.get(Calendar.DAY_OF_YEAR);
        time.set(Calendar.DAY_OF_YEAR, day + 1);
        time.set(Calendar.MONTH, time.get(Calendar.MONTH) + period - 1);
        return new java.sql.Date(time.getTime().getTime());
    }

    public static java.sql.Date periodEndTime(Integer period) {
        Calendar time = Calendar.getInstance();
        int day = time.get(Calendar.DAY_OF_YEAR);
        time.set(Calendar.DAY_OF_YEAR, day + 1);
        time.set(Calendar.MONTH, time.get(Calendar.MONTH) + period);
        return new java.sql.Date(time.getTime().getTime());
    }

    public static String getCurrentDateTime(String formatType) {
        java.util.Date date = new java.util.Date();
        SimpleDateFormat sdf = new SimpleDateFormat(formatType);
        return sdf.format(date);
    }

    /**
     * 返回两个日期之间的天数，每月为30天，每年为360天
     *
     * @param start 开始日期
     * @param end   结束日期
     * @return 间隔的天数
     */
    public static int getDateInterval(java.sql.Date start, java.sql.Date end) {
        GregorianCalendar startCal = new GregorianCalendar();
        GregorianCalendar endCal = new GregorianCalendar();
        startCal.setTime(start);
        endCal.setTime(end);
        int startYear = startCal.get(Calendar.YEAR);
        int endYear = endCal.get(Calendar.YEAR);
        int startMonth = startCal.get(Calendar.MONTH);
        int endMonth = endCal.get(Calendar.MONTH);
        int months = 0;
        if (startYear == endYear) {
            months = endMonth - startMonth;
        } else {
            months = 12 * (endYear - startYear) + endMonth - startMonth;//两个日期相差几个月，即月份差
        }
        // months = months == 0 ? 0 : months - 1;
        int startDay = startCal.get(Calendar.DATE) == 31 ? 30 : startCal.get(Calendar.DATE);
        int endDay = endCal.get(Calendar.DATE) == 31 ? 30 : endCal.get(Calendar.DATE);
        int gap = months * 30 + endDay - startDay;
        return gap;
    }

    public static int dateDiff(String start, String end) {
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss", new Locale("zh", "cn"));
        try {
            long t1 = simpleDateFormat.parse(start).getTime();
            long t2 = simpleDateFormat.parse(end).getTime();
            int diff = (int) ((t2 - t1) / 1000);
            return diff;
        } catch (Exception e) {
            e.printStackTrace();
            return 0;
        }
    }

    /**
     * 两个时间相差距离多少分
     *
     * @param d1 时间参数 1
     * @param d2 时间参数 2
     * @return long 返回值为：分
     */
    public static long getTimeIntervalInMinute(java.util.Date d1, java.util.Date d2) {
        return Math.abs(d2.getTime() - d1.getTime()) / (1000 * 60);
    }

    public static boolean isValidDate(String str) {
        String patternStr =
                "^((((19|20)(([02468][048])|([13579][26]))\\-02\\-29))|((20[0-9][0-9])|(19[0-9][0-9]))\\-((((0[1-9])|(1[0-2]))\\-((0[1-9])|(1\\d)|(2[0-8])))|((((0[13578])|(1[02]))\\-31)|(((01,3-9])|(1[0-2]))\\-(29|30)))))$";
        Pattern p = null;
        Matcher m = null;
        p = Pattern.compile(patternStr);
        m = p.matcher(str);
        boolean b = m.matches();
        if (b) {
            return true;
        }else{
            return false;
        }
    }

    public static String dateDiffForTime(Long endTimeSec, Long startTimeSec) {

        long nd = 1000 * 24 * 60 * 60;//一天的毫秒数
        long nh = 1000 * 60 * 60;//一小时的毫秒数
        long nm = 1000 * 60;//一分钟的毫秒数
        long ns = 1000;//一秒钟的毫秒数
        long diff;
        long day = 0L;//计算差多少天
        long hour = 0L;//计算差多少小时
        long min = 0L;//计算差多少分钟
        long sec = 0L;//计算差多少秒
        StringBuffer result = new StringBuffer();

        //获得两个时间的毫秒时间差异
        diff = endTimeSec - startTimeSec;
        if (diff >= 0) {
            day = diff / nd;//计算差多少天
            hour = diff % nd / nh;//计算差多少小时
            min = diff % nd % nh / nm;//计算差多少分钟
            sec = diff % nd % nh % nm / ns;//计算差多少秒

            result.append(day + "天");
            if (hour < 10) {
                result.append("0" + hour + "小时");
            } else {
                result.append(hour + "小时");
            }
            if (min < 10) {
                result.append("0" + min + "分");
            } else {
                result.append(min + "分");
            }
            if (sec < 10) {
                result.append("0" + sec + "秒");
            } else {
                result.append(sec + "秒");
            }
        } else {
            result.append("0天00小时00分00秒");
        }
        //输出结果
        return result.toString();
    }

    public static String dateDiffForTimeValue(Long endTimeSec, Long startTimeSec) {

        long nd = 1000 * 24 * 60 * 60;//一天的毫秒数
        long nh = 1000 * 60 * 60;//一小时的毫秒数
        long nm = 1000 * 60;//一分钟的毫秒数
        long ns = 1000;//一秒钟的毫秒数
        long diff;
        long day = 0L;//计算差多少天
        long hour = 0L;//计算差多少小时
        long min = 0L;//计算差多少分钟
        long sec = 0L;//计算差多少秒
        StringBuffer result = new StringBuffer();

        //获得两个时间的毫秒时间差异
        diff = endTimeSec - startTimeSec;
        if (diff >= 0) {
            day = diff / nd;//计算差多少天
            hour = diff % nd / nh;//计算差多少小时
            min = diff % nd % nh / nm;//计算差多少分钟
            sec = diff % nd % nh % nm / ns;//计算差多少秒

            result.append(day + ":");
            result.append(hour + ":");
            result.append(min + ":");
            result.append(sec);

        } else {
            result.append("0:0:0:0");
        }
        //输出结果
        return result.toString();
    }

    public static String dateDiffForUsedTime(Long endTimeSec, Long startTimeSec) {

        long nd = 1000 * 24 * 60 * 60;//一天的毫秒数
        long nh = 1000 * 60 * 60;//一小时的毫秒数
        long nm = 1000 * 60;//一分钟的毫秒数
        long ns = 1000;//一秒钟的毫秒数
        long diff;
        long day = 0L;//计算差多少天
        long hour = 0L;//计算差多少小时
        long min = 0L;//计算差多少分钟
        long sec = 0L;//计算差多少秒
        StringBuffer result = new StringBuffer();

        //获得两个时间的毫秒时间差异
        diff = endTimeSec - startTimeSec;
        if (diff >= 0) {
            day = diff / nd;//计算差多少天
            hour = diff % nd / nh;//计算差多少小时
            min = diff % nd % nh / nm;//计算差多少分钟
            sec = diff % nd % nh % nm / ns;//计算差多少秒
            if (day > 0) {
                result.append(day + "天");
            }
            if (hour > 0) {
                result.append(hour + "小时");
            }
            if (min > 0) {
                result.append(min + "分");
            }
            if (sec > 0) {
                result.append(sec + "秒");
            }
        }
        //输出结果
        return result.toString();
    }

    public static String millisecondsToTime(long milliseconds) {

        long nd = 1000 * 24 * 60 * 60;//一天的毫秒数
        long nh = 1000 * 60 * 60;//一小时的毫秒数
        long nm = 1000 * 60;//一分钟的毫秒数
        long ns = 1000;//一秒钟的毫秒数

        long day = 0L;//计算差多少天
        long hour = 0L;//计算差多少小时
        long min = 0L;//计算差多少分钟
        long sec = 0L;//计算差多少秒

        day = milliseconds / nd;//计算差多少天
        hour = milliseconds % nd / nh;//计算差多少小时
        min = milliseconds % nd % nh / nm;//计算差多少分钟
        sec = milliseconds % nd % nh % nm / ns;//计算差多少秒

        return String.format("%02d天%02d时%02d分%02d秒", day, hour, min, sec);
    }

    public static int compareDate(String startDate, String endDate, String formatter) {
        int compareResult = 0;
        DateFormat dateFormat = new SimpleDateFormat(formatter);

        try {
            java.util.Date dateTime1 = dateFormat.parse(startDate);
            java.util.Date dateTime2 = dateFormat.parse(endDate);
            compareResult = dateTime1.compareTo(dateTime2);
        } catch (Exception e) {
            e.printStackTrace();
        }

        return compareResult;
    }

    public static java.sql.Date addDay(java.sql.Date date, int dayNum) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.DATE, dayNum);// 把日期往后增加dayNum天,整数往后推,负数往前移动
        return new java.sql.Date(calendar.getTime().getTime());
    }

    public static java.sql.Date addSecond(java.sql.Date date, int secNum) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.SECOND, secNum);// 把日期往后增加secNum秒,整数往后推,负数往前移动
        return new java.sql.Date(calendar.getTime().getTime());
    }

    public static String getChDateMD(String date) throws Exception {
        StringBuffer out = new StringBuffer("");
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(string2Date(date));
        out.append((calendar.get(Calendar.MONTH) + 1) + "月" + calendar.get(Calendar.DATE) + "日");
        return out.toString();
    }

    public static String getChDateMD(java.sql.Date date) {
        StringBuffer out = new StringBuffer("");
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        out.append((calendar.get(Calendar.MONTH) + 1) + "月" + calendar.get(Calendar.DATE) + "日");
        return out.toString();
    }

    public static String getChDateYMD(String date) throws Exception {
        StringBuffer out = new StringBuffer("");
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(string2Date(date));
        out.append(calendar.get(Calendar.YEAR) + " 年  " + (calendar.get(Calendar.MONTH) + 1) + " 月  " + calendar.get(Calendar.DATE) + " 日  ");
        return out.toString();
    }

    public static String getChDateYMD(java.sql.Date date) throws Exception {
        StringBuffer out = new StringBuffer("");
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        out.append(calendar.get(Calendar.YEAR) + " 年  " + (calendar.get(Calendar.MONTH) + 1) + " 月  " + calendar.get(Calendar.DATE) + " 日  ");
        return out.toString();
    }

    public static int getWeekDay(java.sql.Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        return cal.get(Calendar.DAY_OF_WEEK);
    }

    public static int getHour(java.sql.Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        return cal.get(Calendar.HOUR_OF_DAY);
    }

    public static String date2StringDateTimeNoLine(Timestamp date) {
        if (date == null) {
            return "";
        }
        DateFormat df = new SimpleDateFormat("yyyyMMddHHmmss");
        return df.format(date);
    }

    /**
     * 增添天数的23.59.59S
     *
     * @param date
     * @param dayNum
     * @return
     * @throws ParseException
     */
    public static java.util.Date addDayHms(java.util.Date date, int dayNum) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.DATE, dayNum);
        calendar.set(calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH), calendar.get(Calendar.DAY_OF_MONTH), 23, 59, 59);
        return calendar.getTime();
    }

    /**
     * 获取下一个纪念日
     * @param commemorationDay  “yyyy-MM-dd”
     * @return yyyy-MM-dd
     */
    public static String nextCommemorationDay(String commemorationDay) {
        SimpleDateFormat myFormatter = new SimpleDateFormat(FORMAT_SHORT);
        Calendar cToday = Calendar.getInstance(); // 存今天
        Calendar cBirth = Calendar.getInstance(); // 存纪念日
        Calendar nextBirthDay = Calendar.getInstance();  //存下一个纪念日
        try {
            cBirth.setTime(myFormatter.parse(commemorationDay)); // 设置纪念日
        } catch (ParseException e) {
            e.printStackTrace();
        }

        int month = cBirth.get(Calendar.MONTH);
        int day = cBirth.get(Calendar.DAY_OF_MONTH);

//        int currentYear = cToday.get(Calendar.YEAR);
//        System.out.println("currentYear:"+currentYear+"  month:"+month+"  day:"+day);

        if(month==1 && day==29){
            int curYear = cToday.get(Calendar.YEAR);
            int v = curYear%4;

            int nextCYear = curYear+4-v;

            if(v==0){ //当前年就是闰年
                nextBirthDay.set(Calendar.MONTH, cBirth.get(Calendar.MONTH)); // will修改为本年纪念日
                nextBirthDay.set(Calendar.DAY_OF_MONTH, cBirth.get(Calendar.DAY_OF_MONTH)); // will修改为本年纪念日
                if(nextBirthDay.before(cToday)){  //今年生日已过
                    nextBirthDay.set(Calendar.YEAR,nextCYear);
                }

            }else{ // 当年不是闰年
                nextBirthDay.set(Calendar.YEAR,nextCYear);
                nextBirthDay.set(Calendar.MONTH, cBirth.get(Calendar.MONTH)); // will修改为本年纪念日
                nextBirthDay.set(Calendar.DAY_OF_MONTH, cBirth.get(Calendar.DAY_OF_MONTH)); // will修改为本年纪念日
            }


        }else {
            nextBirthDay.set(Calendar.MONTH, cBirth.get(Calendar.MONTH)); // will修改为本年纪念日
            nextBirthDay.set(Calendar.DAY_OF_MONTH, cBirth.get(Calendar.DAY_OF_MONTH)); // will修改为本年纪念日
            if(nextBirthDay.before(cToday)){  //今年纪念日已过
                nextBirthDay.add(Calendar.YEAR,1);
            }
        }

        return myFormatter.format(nextBirthDay.getTime());
    }

    /*
     * 将时间转换为时间戳
     */
    public static int dateToStamp(String dt){
        String res;
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        Date date = null;
        try {
            date = simpleDateFormat.parse(dt);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        int ts = (int)(date.getTime()/1000);
        return ts;
    }
    /**
     * 获取下个月的第一秒
     * @param timestamp
     * @return
     */
    public static long getNextMonth1stTimestamp(long timestamp) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date(timestamp));
        calendar.add(Calendar.MONTH, 1);
        calendar.set(Calendar.DAY_OF_MONTH,1);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        return calendar.getTime().getTime();
    }
    /**
     * 获取上个月的第一秒
     * @param timestamp
     * @return
     */
    public static long getLastMonth1stTimestamp(long timestamp) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date(timestamp));
        calendar.add(Calendar.MONTH, -1);
        calendar.set(Calendar.DAY_OF_MONTH,1);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        return calendar.getTime().getTime();
    }
    /**
     * 获取当前月的最后一秒
     * @param timestamp
     * @return
     */
    public static long getMonthLastTimestamp(long timestamp) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date(timestamp));
        calendar.set(Calendar.DAY_OF_MONTH,1);
        calendar.roll(Calendar.DATE, -1);
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        return calendar.getTime().getTime();
    }
    /**
     * 获取当前月的第一秒
     * @param timestamp
     * @return
     */
    public static long getMonth1stTimestamp(long timestamp) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date(timestamp));
        calendar.set(Calendar.DAY_OF_MONTH,1);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        return calendar.getTime().getTime();
    }

    public static void main(String[] args) throws ParseException {



        //--------------------------------------------------------------------------------

//        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
//
//        Date d1 = parse("2019-05-23 23:00:01","yyyy-MM-dd");
//        Date d2 = new Date(); // parse("2019-05-18 06:00:00","yyyy-MM-dd");
//
////        String d2_str = dateFormat.format(d2);
////        String d1_str = dateFormat.format(d1);
////
////        Date d1New = parse(d1_str,"yyyy-MM-dd");
////        Date d2New = parse(d2_str,"yyyy-MM-dd");
//
//
//        Date d1New = parseYMD(d1);
//        Date d2New = parseYMD(d2);
//
//
//        System.out.println(d2New.compareTo(d1New));


        //--------------------------------------------------------------------------

//        SimpleDateFormat myFormatter = new SimpleDateFormat("yyyy-MM-dd");
//        String clidate = "1996-02-29";
//        Calendar cToday = Calendar.getInstance(); // 存今天
//        Calendar cBirth = Calendar.getInstance(); // 存生日
//        Calendar nextBirthDay = Calendar.getInstance();  //存未来生日
//        cBirth.setTime(myFormatter.parse(clidate)); // 设置生日
//
//        int month = cBirth.get(Calendar.MONTH);
//        int day = cBirth.get(Calendar.DAY_OF_MONTH);
//
//        int currentYear = cToday.get(Calendar.YEAR);
//        System.out.println("currentYear:"+currentYear+"  month:"+month+"  day:"+day);
//
//        if(month==1 && day==29){
//            System.out.println("1--->1");
//            int curYear = cToday.get(Calendar.YEAR);
//            int v = curYear%4;
//
//            int nextCYear = curYear+4-v;
//
//            if(v==0){ //当前年就是闰年
//                System.out.println("1--->2");
//                nextBirthDay.set(Calendar.MONTH, cBirth.get(Calendar.MONTH)); // will修改为本年生日
//                nextBirthDay.set(Calendar.DAY_OF_MONTH, cBirth.get(Calendar.DAY_OF_MONTH)); // will修改为本年生日
//                if(nextBirthDay.before(cToday)){  //今年生日已过
//                    nextBirthDay.set(Calendar.YEAR,nextCYear);
//                }
//
//            }else{ // 当年不是闰年
//                System.out.println("1--->3");
//                nextBirthDay.set(Calendar.YEAR,nextCYear);
//                nextBirthDay.set(Calendar.MONTH, cBirth.get(Calendar.MONTH)); // will修改为本年生日
//                nextBirthDay.set(Calendar.DAY_OF_MONTH, cBirth.get(Calendar.DAY_OF_MONTH)); // will修改为本年生日
//            }
//
//        }else {
//            System.out.println("2--->1");
//            nextBirthDay.set(Calendar.MONTH, cBirth.get(Calendar.MONTH)); // will修改为本年生日
//            nextBirthDay.set(Calendar.DAY_OF_MONTH, cBirth.get(Calendar.DAY_OF_MONTH)); // will修改为本年生日
//            if(nextBirthDay.before(cToday)){  //今年生日已过
//                System.out.println("2--->2");
//                nextBirthDay.add(Calendar.YEAR,1);
//            }
//        }
//
//        System.out.println("today: "+myFormatter.format(cToday.getTime()));
//        System.out.println("birth: "+myFormatter.format(cBirth.getTime()));
//        System.out.println("nextBirthday: "+myFormatter.format(nextBirthDay.getTime()));


        String day = "2019-05-24";

        String result = nextCommemorationDay(day);
        System.out.println(result);

//        Time time = new Time(12,25,30);
//
//        String timeStr = time.toString();
//
//        System.out.println(timeStr);

    }
}
