package com.lanren.huhu.partner.schedule;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.lanren.huhu.partner.constants.Constants;
import com.lanren.huhu.partner.domain.*;
import com.lanren.huhu.partner.model.*;
import com.lanren.huhu.partner.service.*;
import com.lanren.huhu.partner.util.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;

import static com.lanren.huhu.partner.constants.Constants.NEW_SETTLE_RULE_TIME;
import static org.springframework.util.StringUtils.isEmpty;

/**
 * @author chen
 * @title: PartnerSettleTask
 * @projectName partner
 * @description: 合伙人结算
 * @package com.lanren.huhu.partner.schedule
 * @date 2019-07-02 15:45
 */
@Component
public class PartnerSettleTask {
    private static Logger logger = LoggerFactory.getLogger(PartnerSettleTask.class);
    private static final int PAGE_SIZE = 1000;
    private static final BigDecimal ZERO = new BigDecimal(0);

    @Autowired
    PartnerRewardService partnerRewardService;
    @Autowired
    OrdersActiveService ordersActiveService;
    @Autowired
    OrderTaobaoJdCommissionService orderTaobaoJdCommissionService;
    @Autowired
    OrderTaobaoJdGoodsCommissionService orderTaobaoJdGoodsCommissionService;
    @Autowired
    DataSourceTransactionManager dataSourceTransactionManager;
    @Autowired
    TransactionDefinition transactionDefinition;
    @Autowired
    PartnerAccountService partnerAccountService;
    @Autowired
    PartnerAccountLogService partnerAccountLogService;
    @Autowired
    PartnerIncomeSummaryService partnerIncomeSummaryService;

    @Async
    public void runSettle() {
        logger.info("run PartnerSettleTask");
        try {
            long input = DateUtils.getLastMonth1stTimestamp(System.currentTimeMillis());
            long beginTs = DateUtils.getMonth1stTimestamp(input) / 1000L;
            long endTs  = DateUtils.getMonthLastTimestamp(input) / 1000L;
            updateAll(beginTs, endTs);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
        logger.info("run PartnerSettleTask done");
    }

    private void updateAll(long beginTs, long endTs) {
        updateOrderCommission(beginTs, endTs, 0, 0);
        doSettle(beginTs, endTs);
    }

    public boolean doOneUserSettle(Integer userId) {
        long input = DateUtils.getLastMonth1stTimestamp(System.currentTimeMillis());
        long beginTs = DateUtils.getMonth1stTimestamp(input) / 1000L;
        long endTs  = DateUtils.getMonthLastTimestamp(input) / 1000L;
        PartnerSettle errPartnerSettle = null;
        try {
            updateOrderCommission(beginTs, endTs, 1, userId);
            PartnerSettle partnerSettle = partnerRewardService.selectPartnerSettleByTsAndUid(beginTs, endTs, userId);
            errPartnerSettle = partnerSettle;
            if (partnerSettle != null && setIncomeAndTax(partnerSettle)) {
                logger.info("partner settle is {}", partnerSettle.toString());
                return doDbUpdate(partnerSettle, beginTs, endTs);
            } else {
                logger.info("没有待奖励记录 跳过不处理, 用户{}", userId);
                return true;
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            logger.error("userId is {}, errPartnerSettle is {}", userId, errPartnerSettle);
            return false;
        }
    }

    /**
     * 使用mybatisplus分页插件时需要注意:
     * !!! 如果取出每一页的数据后, 在处理时更改了分页查询的一些字段, 那么后面翻页的时候 查出来的数据就会错乱了
     * !!! 要么就是 在循环的时候不更新用于查询的字段
     * !!! 要么就是 永远都只取第一页, 直到取不到数据
     */
    private void doSettle(long beginTs, long endTs) {
        PartnerSettle errPartnerSettle = null;
        try {
            int cnt = 0;
            while (true) {
                int pageNo = 1;
                Page<PartnerSettle> page = new Page<PartnerSettle>(pageNo, PAGE_SIZE);
                page.setOptimizeCountSql(false);
                page.setSearchCount(false);
                IPage<PartnerSettle> ipage = partnerRewardService.selectPartnerSettlePageByTs(page, beginTs, endTs);
                List<PartnerSettle> partnerSettleList = ipage.getRecords();
                if (partnerSettleList != null && partnerSettleList.size() > 0) {
                    for (PartnerSettle partnerSettle : partnerSettleList) {
                        try {
                            errPartnerSettle = partnerSettle;
                            if (partnerSettle != null && setIncomeAndTax(partnerSettle)) {
                                if (!doDbUpdate(partnerSettle, beginTs, endTs)) {
                                    logger.error("doDbUpdate failed for user {}", partnerSettle.getUserId());
                                }
                            }
                        } catch (Exception e) {
                            logger.error(e.getMessage(), e);
                        }
                        cnt++;
                    }
                } else {
                    break;
                }
            }
            logger.info("cnt is {}", cnt);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            logger.error("errPartnerSettle is {}", errPartnerSettle);
        }
    }

    private boolean doDbUpdate(PartnerSettle partnerSettle, long beginTs, long endTs) {
        TransactionStatus transactionStatus = null;
        try {
            logger.info("partnerSettle is {}", partnerSettle);
            /**
             * 开启事务
             */
            transactionStatus = dataSourceTransactionManager.getTransaction(transactionDefinition);

            /**
             * 获取合伙人账户 锁定行
             */
            PartnerAccount partnerAccount = partnerAccountService.getOneByUserIdForUpdate(partnerSettle.getUserId());
            /**
             * 插入变更记录
             */
            insertIntoPartnerAccountLog(partnerSettle, partnerAccount, beginTs);
            updatePartnerAccount(partnerSettle, partnerAccount);
            updatePartnerRewardStatus(partnerSettle.getUserId(), beginTs, endTs);
            insertPartnerIncomeSummary(partnerSettle, partnerAccount, beginTs);
            /**
             * 提交事务
             */
            dataSourceTransactionManager.commit(transactionStatus);
//            dataSourceTransactionManager.rollback(transactionStatus);
            return true;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            /**
             * 回滚事务
             */
            if (transactionStatus != null) {
                logger.info("do settle failed, rollback");
                dataSourceTransactionManager.rollback(transactionStatus);
            }
            return false;
        }
    }

    private void insertPartnerIncomeSummary(PartnerSettle partnerSettle,PartnerAccount partnerAccount, long beginTs) {
        Date dat = new Date(beginTs * 1000L);
        int now = (int) (System.currentTimeMillis() / 1000L);
        int yeartime = Integer.parseInt(DateUtils.getYear(dat));
        int monthtime = Integer.parseInt(DateUtils.getMonth(dat));
        PartnerIncomeSummary partnerIncomeSummary = partnerIncomeSummaryService.getOneByUidYearMonth(partnerSettle.getUserId(), yeartime, monthtime);
        /**
         * 生成content字段
         */
        PartnerSummaryContent partnerSummaryContent = new PartnerSummaryContent();
        partnerSummaryContent.setCash(partnerSettle.getCash().setScale(2, RoundingMode.HALF_UP));
        partnerSummaryContent.setUpCash(partnerSettle.getUpCash().setScale(2, RoundingMode.HALF_UP));
        partnerSummaryContent.setIcash(partnerSettle.getIcash().setScale(2, RoundingMode.HALF_UP));
        partnerSummaryContent.setPcash(partnerSettle.getPcash().setScale(2, RoundingMode.HALF_UP));
        partnerSummaryContent.setScash(partnerSettle.getScash().setScale(2, RoundingMode.HALF_UP));
        partnerSummaryContent.setRcash(partnerSettle.getRcash().setScale(2, RoundingMode.HALF_UP));

        partnerSummaryContent.setTechChargeOrder(partnerSettle.getTechChargeOrder());
        partnerSummaryContent.setTechChargeRedpack(partnerSettle.getTechChargeRedpack());
        partnerSummaryContent.setWxIncome(partnerSettle.getWxIncome());
        partnerSummaryContent.setWxTax(partnerSettle.getWxTax());
        partnerSummaryContent.setBankIncome(partnerSettle.getBankIncome());
        partnerSummaryContent.setBankTax(partnerSettle.getBankTax());
        partnerSummaryContent.setWxFreeBalance(partnerAccount.getWxFreeBalance());
        partnerSummaryContent.setBankFreeBalance(partnerAccount.getBankFreeBalance());

        partnerSummaryContent.setSumOrderCommission(partnerSettle.getSumOrderCommission());
        partnerSummaryContent.setSumRedpack(partnerSettle.getSumRedpack());
        partnerSummaryContent.setSumOthers(partnerSettle.getSumOthers());
        /**
         * 生成income字段
         */
        BigDecimal sum1 = partnerSettle.getCash()
                .add(partnerSettle.getUpCash())
                .add(partnerSettle.getIcash())
                .add(partnerSettle.getPcash())
                .add(partnerSettle.getScash())
                .add(partnerSettle.getRcash());
        BigDecimal sum2 = partnerSettle.getSumOrderCommission()
                .add(partnerSettle.getSumRedpack())
                .add(partnerSettle.getSumOthers())
                .add(partnerSettle.getTechChargeOrder())
                .add(partnerSettle.getTechChargeRedpack());
        if (sum1.compareTo(sum2) != 0) {
            logger.error("合伙人奖励结算-对账异常: 求和1:{}, 求和2:{}, 奖励字段:{}", sum1, sum2, partnerSettle);
        }
        if (null == partnerIncomeSummary) {
            partnerIncomeSummary = new PartnerIncomeSummary();
            partnerIncomeSummary.setUserId(partnerSettle.getUserId());
            partnerIncomeSummary.setYeartime(yeartime);
            partnerIncomeSummary.setMonthtime(monthtime);
            partnerIncomeSummary.setState(20);
            partnerIncomeSummary.setCreatedAt(now);
            partnerIncomeSummary.setUpdatedAt(now);
            partnerIncomeSummary.setSettleTime(now);
            partnerIncomeSummary.setContent(JSON.toJSONString(partnerSummaryContent).replaceAll("0E-[0-9]+", "0"));
            partnerIncomeSummary.setIncome(sum2);
            partnerIncomeSummaryService.save(partnerIncomeSummary);
        } else {
            partnerIncomeSummary.setState(20);
            partnerIncomeSummary.setUpdatedAt(now);
            partnerIncomeSummary.setSettleTime(now);
            partnerIncomeSummary.setContent(JSON.toJSONString(partnerSummaryContent).replaceAll("0E-[0-9]+", "0"));
            partnerIncomeSummary.setIncome(sum2);
            partnerIncomeSummaryService.updateById(partnerIncomeSummary);
        }
    }

    private int updatePartnerRewardStatus(int userId, long beginTs, long endTs) {
        return partnerRewardService.updateRewardStatusByUidAndTs(userId, beginTs, endTs);
    }

    private void updatePartnerAccount(PartnerSettle partnerSettle, PartnerAccount partnerAccount) {
        /**
         * 更新合伙人账户金额
         */
        BigDecimal wxIncomeAdd = partnerSettle.getWxIncome();
        BigDecimal bankIncomeAdd = partnerSettle.getBankIncome();
        partnerAccount.setAllIncome(partnerAccount.getAllIncome().add(wxIncomeAdd).add(bankIncomeAdd));
        partnerAccount.setIncomeAfterTax(partnerAccount.getIncomeAfterTax().add(wxIncomeAdd).add(bankIncomeAdd));
        partnerAccount.setWxFreeBalance(partnerAccount.getWxFreeBalance().add(wxIncomeAdd));
        partnerAccount.setBankFreeBalance(partnerAccount.getBankFreeBalance().add(bankIncomeAdd));
        partnerAccountService.updateById(partnerAccount);
    }

    private void insertIntoPartnerAccountLog(PartnerSettle partnerSettle, PartnerAccount partnerAccount, long beginTs) {
        Date dat = new Date(beginTs * 1000L);
        /**
         * 月结发钱
         */
        BigDecimal changeNum = partnerSettle.getSumOrderCommission()
                .add(partnerSettle.getSumRedpack())
                .add(partnerSettle.getSumOthers())
                .add(partnerSettle.getTechChargeRedpack())
                .add(partnerSettle.getTechChargeOrder());
        BigDecimal changedAmount = partnerAccount.getWxFreeBalance().add(partnerAccount.getBankFreeBalance()).add(changeNum);
        PartnerSnapshot snapshot = new PartnerSnapshot();
        snapshot.setUserId(partnerAccount.getUserId());
        snapshot.setWxFreeBalance(partnerAccount.getWxFreeBalance());
        snapshot.setBankFreeBalance(partnerAccount.getBankFreeBalance());
        PartnerAccountLog log = createNewPartnerAccountLog(partnerSettle.getUserId(),
                Constants.PARTNER_ACCOUNT_LOG_TYPE_SETTLE,
                changeNum,
                changedAmount,
                DateUtils.format(dat, "yyyy-MM") + Constants.PARTNER_ACCOUNT_LOG_TYPE_SETTLE_REMARK,
                snapshot.toString());
        partnerAccountLogService.save(log);
        /**
         * 平台技术服务费
         */
        changeNum = partnerSettle.getTechChargeRedpack()
                .add(partnerSettle.getTechChargeOrder())
                .negate();
        changedAmount = changedAmount.add(changeNum);
        if (changeNum.compareTo(ZERO) < 0) {
            log = createNewPartnerAccountLog(partnerSettle.getUserId(),
                    Constants.PARTNER_ACCOUNT_LOG_TYPE_TECH_CHARGE,
                    changeNum,
                    changedAmount,
                    DateUtils.format(dat, "yyyy-MM") + Constants.PARTNER_ACCOUNT_LOG_TYPE_TECH_CHARGE_REMARK,
                    "");
            partnerAccountLogService.save(log);
        }
        /**
         * 个人所得税
         */
        changeNum = partnerSettle.getWxTax().add(partnerSettle.getBankTax()).negate();
        changedAmount = changedAmount.add(changeNum);
        snapshot.setWxFreeBalance(snapshot.getWxFreeBalance().add(partnerSettle.getWxIncome()).add(partnerSettle.getWxTax()));
        snapshot.setBankFreeBalance(snapshot.getBankFreeBalance().add(partnerSettle.getBankIncome()).add(partnerSettle.getBankTax()));
        if (changeNum.compareTo(ZERO) < 0) {
            log = createNewPartnerAccountLog(partnerSettle.getUserId(),
                    Constants.PARTNER_ACCOUNT_LOG_TYPE_TAX,
                    changeNum,
                    changedAmount,
                    DateUtils.format(dat, "yyyy-MM") + Constants.PARTNER_ACCOUNT_LOG_TYPE_TAX_REMARK,
                    snapshot.toString());
            partnerAccountLogService.save(log);
        }
    }

    private PartnerAccountLog createNewPartnerAccountLog(int userId, int type, BigDecimal changeNum, BigDecimal changedAmount, String payRemark,String snapshot) {
        Date now = new Date();
        int yeartime = Integer.parseInt(DateUtils.getYear(now));
        int monthtime = Integer.parseInt(DateUtils.getMonth(now));
        int daytime = Integer.parseInt(DateUtils.getDay(now));
        PartnerAccountLog log = new PartnerAccountLog();
        log.setUserId(userId);
        log.setType(type);
        log.setChangeNum(changeNum);
        log.setChangedAmount(changedAmount);
        log.setPayRemark(payRemark);
        log.setPayType(false);
        log.setYeartime(yeartime);
        log.setMonthtime(monthtime);
        log.setDaytime(daytime);
        log.setSnapshot(JSON.toJSONString(snapshot).replaceAll("0E-[0-9]+", "0"));
        log.setPayState(20);
        log.setPayPlatform("hupay");
        log.setPayInfo(payRemark);
        log.setChangeTime((int) (now.getTime() / 1000L));
        return log;
    }

    private void add2UpdateRewardCommission(PartnerReward partnerReward, BigDecimal allMoney, List<PartnerReward> updateCommisionList) {
        try {
            partnerReward.setAllMoneyOri(allMoney);
            String ratioAllStr = partnerReward.getRatioAll().replaceAll("[\\[\\]\"]", "");
            /**
             * 时间点之前的 需要乘比例数组 之后 再乘commission_rate
             * 时间点之后的 直接乘commission_rate
             */
            if (partnerReward.getRechargeTime().compareTo(NEW_SETTLE_RULE_TIME) < 0) {
                if (! isEmpty(ratioAllStr)) {
                    String[] strs = ratioAllStr.split(",");
                    if (strs.length > 0) {
                        BigDecimal cash = allMoney;
                        for (String s : strs) {
                            BigDecimal ratio = new BigDecimal(s);
                            cash = cash.multiply(ratio);
                        }
                        partnerReward.setCash(cash);
                        BigDecimal commission = cash.multiply(partnerReward.getCommissionRate());
                        partnerReward.setCommissionAcount(commission);
                        partnerReward.setSettleState(Constants.SETTLE_STATE_DONE);
                        partnerReward.setUpdatedAt(System.currentTimeMillis() / 1000L);
//                    partnerRewardService.updateById(partnerReward);
                        updateCommisionList.add(partnerReward);
                    }
                }
            } else {
                partnerReward.setCash(allMoney);
                BigDecimal commission = allMoney.multiply(partnerReward.getCommissionRate());
                partnerReward.setCommissionAcount(commission);
                partnerReward.setSettleState(Constants.SETTLE_STATE_DONE);
                partnerReward.setUpdatedAt(System.currentTimeMillis() / 1000L);
//                    partnerRewardService.updateById(partnerReward);
                updateCommisionList.add(partnerReward);
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            logger.error("partnerReward id is {}", partnerReward.getId());
        }
    }
    /**
     * 更新至 失效, 结算完成
     * 不再参与后续结算计算
     * @param partnerReward
     */
    private void add2UpdateReward2Failed(PartnerReward partnerReward, List<PartnerReward> update2FailedList) {
        if (partnerReward.getCommissionAcount().compareTo(BigDecimal.ZERO) >= 0) {
            partnerReward.setCommissionAcount(ZERO);
        }
        partnerReward.setAllMoneyOri(ZERO);
        partnerReward.setCash(ZERO);
        if (partnerReward.getSettleState() > 0) {
            partnerReward.setSettleState(partnerReward.getSettleState() * -1);
        }
        partnerReward.setUpdatedAt(System.currentTimeMillis() / 1000L);
//        partnerRewardService.updateById(partnerReward);
        update2FailedList.add(partnerReward);
    }

    /**
     * 自购省和分享赚的奖励金额 需要根据最终的实际原始佣金 重新计算
     * @param beginTs
     * @param endTs
     */
    private void updateOrderCommission(long beginTs, long endTs, int runType, int userId) {
        PartnerReward errPartnerReward = null;
        try {
            int pageNo = 1;
            int total = 0;
            int totalPage = 0;
            int cnt = 0;
            while (true) {
                Page<PartnerReward> page = new Page<PartnerReward>(pageNo, PAGE_SIZE);
                page.setOptimizeCountSql(false);
                IPage<PartnerReward> ipage;
                if (runType == 0) {
                    ipage = partnerRewardService.selectOrderRewardPageByTs(page, beginTs, endTs);
                } else {
                    ipage = partnerRewardService.selectOrderRewardPageByTsAndUid(page, beginTs, endTs, userId);
                }
                if (total == 0) {
                    total = (int) ipage.getTotal();
                    totalPage = total % PAGE_SIZE == 0 ? total / PAGE_SIZE : total / PAGE_SIZE + 1;
                    logger.info("totalPage is {} ", totalPage);
                }
                List<PartnerReward> partnerRewardList = ipage.getRecords();
                List<PartnerReward> updateCommisionList = new ArrayList<PartnerReward>();
                List<PartnerReward> update2FailedList = new ArrayList<PartnerReward>();
                if (partnerRewardList != null && partnerRewardList.size() > 0) {
                    /**
                     * 加载ordersActiveMap orderMap subOrderMap
                     */
                    Set<OrderParam> orderParamsList = generateOrderParamsSet(partnerRewardList);
                    Set<SubOrderParam> subOrderParamsList = generateSubOrderParamsSet(partnerRewardList);
                    Map<OrderParam, OrdersActive> ordersActiveMap = generateOrdersActiveMap(orderParamsList);
                    Map<OrderParam, OrderTaobaoJdCommission> orderMap = generateOrderMap(orderParamsList);
                    Map<SubOrderParam, OrderTaobaoJdGoodsCommission> subOrderMap = generateSubOrderMap(subOrderParamsList);
                    /**
                     * 循环处理每条奖励
                     */
                    for (PartnerReward partnerReward : partnerRewardList) {
                        try {
                            errPartnerReward = partnerReward;
                            /**
                             * 购物奖励: reward_type: 110, settle_state > 0 且 != 300
                             * 需要根据订单号查找最终的真实的原始佣金 并 按比例快照重新计算奖励金额,
                             * 更新字段: all_money_ori cash commission_account
                             * 如果订单失效: 更新settle_state, all_money_ori/cash/commission_account 抹平, 跳过处理, 循环到下一条
                             */
                            if (partnerReward.getRewardType() == Constants.PARTNER_REWARD_TYPE_PURCHASE && partnerReward.getSettleState() > 0
                                    && partnerReward.getSettleState() != Constants.SETTLE_STATE_DONE) {
                                if (isEmpty(partnerReward.getOrderType()) || isEmpty(partnerReward.getOrderSn())) {
                                    /**
                                     * 历史数据: 没有记录订单号 或者 没有记录比例 不更新奖励金额
                                     */
                                    continue;
                                } else if (partnerReward.getSettleState() < 0) {
                                    continue;
                                } else if (partnerReward.getCommissionAcount().compareTo(BigDecimal.ZERO) == 0) {
                                    /**
                                     * amout为0的 跳过不处理
                                     * 红包兑换的奖励 状态是有效 奖励金额为0
                                     */
                                    continue;
                                } else {
                                    /**
                                     * 有订单号 和 比例快照 可以更新金额
                                     */
//                                    OrdersActive ordersActive = ordersActiveService.getOneByOrderSn(partnerReward.getOrderType(), partnerReward.getOrderSn());
//                                    OrderTaobaoJdCommission order = orderTaobaoJdCommissionService.getOneByOrderSn(partnerReward.getOrderType(), partnerReward.getOrderSn());
                                    OrderParam orderParam = new OrderParam();
                                    orderParam.setOrderType(partnerReward.getOrderType());
                                    orderParam.setOrderSn(partnerReward.getOrderSn());
                                    OrdersActive ordersActive = ordersActiveMap.get(orderParam);
                                    OrderTaobaoJdCommission order = orderMap.get(orderParam);
                                    if (null == ordersActive || null == order) {
                                        /**
                                         * 如果没有找到active 或者 没有找到commission
                                         * 不更新奖励金额
                                         */
                                        continue;
                                    } else {
                                        /**
                                         * 订单状态只有三种: -3, 1, 3
                                         */
                                        if (order.getStatus() == Constants.ORDER_STATE_FAILED) {
                                            /**
                                             * 订单失效 订单付款 跳过, 处理下一条奖励
                                             */
                                            add2UpdateReward2Failed(partnerReward, update2FailedList);
                                        } else if (order.getStatus() == Constants.ORDER_STATE_PAID) {
                                            /**
                                             * 订单还是付款状态 更新奖励时间 下个月再处理
                                             */
                                            long newRechargeTime = DateUtils.getNextMonth1stTimestamp(partnerReward.getRechargeTime().getTime());
                                            partnerReward.setRechargeTime(new Date(newRechargeTime));
                                            partnerReward.setSettleTime(new Date(newRechargeTime));
                                            partnerReward.setUpdatedAt(System.currentTimeMillis() / 1000L);
                                            partnerRewardService.updateById(partnerReward);
                                        } else if (order.getStatus() == Constants.ORDER_STATE_SETTLED) {
                                            /**
                                             * 订单结算 更新金额
                                             */
                                            if (ordersActive.getIsForNew() == 1) {
                                                /**
                                                 * 新人专享订单 不给合伙人奖励 跳过
                                                 */
                                                add2UpdateReward2Failed(partnerReward, update2FailedList);
                                            } else {
                                                /**
                                                 * 正常订单 更新原始佣金
                                                 * all_money_ori: 原始佣金
                                                 * cash: all_money_ori 乘以 ratio_all中的每个比例
                                                 * commission_account: cash 乘以 commission_rate
                                                 */
                                                add2UpdateRewardCommission(partnerReward, order.getAllMoney(), updateCommisionList);
                                            }
                                        }
                                    }
                                }
                            }
                            /**
                             * 分享赚订单: 按子订单号更新奖励金额
                             */
                            if (partnerReward.getRewardType() == Constants.PARTNER_REWARD_TYPE_SHARE && partnerReward.getSettleState() > 0
                                    && partnerReward.getSettleState() != Constants.SETTLE_STATE_DONE) {
                                if (isEmpty(partnerReward.getOrderType()) || isEmpty(partnerReward.getOrderSn()) || isEmpty(partnerReward.getSubOrderSn())) {
                                    /**
                                     * 历史数据: 没有记录子订单号 或者 没有记录比例 不更新奖励金额
                                     */
                                    continue;
                                } else if (partnerReward.getSettleState() < 0) {
                                    continue;
                                } else if (partnerReward.getCommissionAcount().compareTo(BigDecimal.ZERO) == 0) {
                                    /**
                                     * amout为0的 跳过不处理
                                     * 红包兑换的奖励 状态是有效 奖励金额为0
                                     */
                                    continue;
                                } else {
                                    /**
                                     * 有订单号 和 比例快照 可以更新金额
                                     */
                                    SubOrderParam subOrderParam = new SubOrderParam();
                                    subOrderParam.setOrderType(partnerReward.getOrderType());
                                    subOrderParam.setOrderSn(partnerReward.getOrderSn());
                                    subOrderParam.setSubOrderSn(partnerReward.getSubOrderSn());
                                    OrderTaobaoJdGoodsCommission subOrder = subOrderMap.get(subOrderParam);
                                    if (null == subOrder) {
                                        /**
                                         * 如果没有找到子订单
                                         * 不更新奖励金额
                                         */
                                        continue;
                                    } else {
                                        /**
                                         * 订单状态只有三种: -3, 1, 3
                                         */
                                        if (subOrder.getOrderState() == Constants.ORDER_STATE_FAILED) {
                                            /**
                                             * 订单失效 订单付款 跳过, 处理下一条奖励
                                             */
                                            add2UpdateReward2Failed(partnerReward, update2FailedList);
                                        } else if (subOrder.getOrderState() == Constants.ORDER_STATE_PAID) {
                                            /**
                                             * 订单还是付款状态 更新奖励时间 下个月再处理
                                             */
                                            long newRechargeTime = DateUtils.getNextMonth1stTimestamp(partnerReward.getRechargeTime().getTime());
                                            partnerReward.setRechargeTime(new Date(newRechargeTime));
                                            partnerReward.setSettleTime(new Date(newRechargeTime));
                                            partnerReward.setUpdatedAt(System.currentTimeMillis() / 1000L);
                                            partnerRewardService.updateById(partnerReward);
                                        } else if (subOrder.getOrderState() == Constants.ORDER_STATE_SETTLED) {
                                            /**
                                             * 正常订单 更新原始佣金
                                             * all_money_ori: 原始佣金
                                             * cash: all_money_ori 乘以 ratio_all中的每个比例
                                             * commission_account: cash 乘以 commission_rate
                                             */
                                            add2UpdateRewardCommission(partnerReward, subOrder.getAllMoney(), updateCommisionList);
                                        }
                                    }
                                }
                            }
                        } catch (Exception e) {
                            logger.error(e.getMessage(), e);
                        }
                        cnt++;
                    }
                }
                if (updateCommisionList.size() > 0) {
                    partnerRewardService.updateBatch(updateCommisionList);
                }
                if (update2FailedList.size() > 0) {
                    partnerRewardService.updateBatch(update2FailedList);
                }
                if (pageNo >= totalPage) {
                    break;
                }
                pageNo++;
            }
            logger.info("cnt is {}", cnt);
        } catch (Exception e) {
            logger.error(e.getMessage(),e);
            logger.error("errPartnerReward is {}", errPartnerReward);
        }
    }

    private Set<OrderParam> generateOrderParamsSet(List<PartnerReward> partnerRewardList) {
        Set<OrderParam> set = new HashSet<OrderParam>();
        for (PartnerReward reward : partnerRewardList) {
            OrderParam orderParam = new OrderParam();
            orderParam.setOrderType(reward.getOrderType());
            orderParam.setOrderSn(reward.getOrderSn());
            set.add(orderParam);
        }
        return set;
    }

    private Set<SubOrderParam> generateSubOrderParamsSet(List<PartnerReward> partnerRewardList) {
        Set<SubOrderParam> set = new HashSet<SubOrderParam>();
        for (PartnerReward reward : partnerRewardList) {
            SubOrderParam orderParam = new SubOrderParam();
            orderParam.setOrderType(reward.getOrderType());
            orderParam.setOrderSn(reward.getOrderSn());
            orderParam.setSubOrderSn(reward.getSubOrderSn());
            set.add(orderParam);
        }
        return set;
    }


    private Map<SubOrderParam, OrderTaobaoJdGoodsCommission> generateSubOrderMap(Set<SubOrderParam> subOrderParamsSet) {
        Map<SubOrderParam, OrderTaobaoJdGoodsCommission> map = new HashMap<SubOrderParam, OrderTaobaoJdGoodsCommission>();
        List<OrderTaobaoJdGoodsCommission> list = orderTaobaoJdGoodsCommissionService.selectListBySubOrderParamsSet(subOrderParamsSet);
        for (OrderTaobaoJdGoodsCommission subOrder : list) {
            SubOrderParam key = new SubOrderParam();
            key.setOrderType(subOrder.getType());
            key.setOrderSn(subOrder.getOrderSn());
            key.setSubOrderSn(subOrder.getSubOrderSn());
            map.put(key, subOrder);
        }
        return map;
    }

    private Map<OrderParam, OrderTaobaoJdCommission> generateOrderMap(Set<OrderParam> orderParamSet) {
        Map<OrderParam, OrderTaobaoJdCommission> map = new HashMap<OrderParam, OrderTaobaoJdCommission>();
        List<OrderTaobaoJdCommission> list = orderTaobaoJdCommissionService.selectListByOrderParamSet(orderParamSet);
        for (OrderTaobaoJdCommission order : list) {
            OrderParam key = new OrderParam();
            key.setOrderType(order.getType());
            key.setOrderSn(order.getOrderSn());
            map.put(key, order);
        }
        return map;
    }

    private Map<OrderParam, OrdersActive> generateOrdersActiveMap(Set<OrderParam> orderParamSet) {
        Map<OrderParam, OrdersActive> map = new HashMap<OrderParam, OrdersActive>();
        List<OrdersActive> list = ordersActiveService.selectListByOrderParamSet(orderParamSet);
        for (OrdersActive order : list) {
            OrderParam key = new OrderParam();
            key.setOrderType(order.getType());
            key.setOrderSn(order.getOrderSn());
            map.put(key, order);
        }
        return map;
    }

    private static boolean setIncomeAndTax(PartnerSettle partnerSettle) {
        try {
            /**
             * base为算税部分收入
             */
            BigDecimal base = partnerSettle.getSumRedpack().add(partnerSettle.getSumOthers());
            /**
             * 订单收入全部计入微信 并且不计税
             * wxIncome 和 bankIncome 均为扣税后的收入
             */
            BigDecimal wxIncome = partnerSettle.getSumOrderCommission();
            BigDecimal taxRate = BigDecimal.valueOf(0.075).divide(BigDecimal.valueOf(1.075));
            BigDecimal wxTax;
            /**
             * 大于800的部分 按 乘以系数 ( 0.075 / (1 + 0.075) ) 算税
             */
            if (base.compareTo(Constants.TAX_STAGE_FREE_BASE) <= 0) {
                wxTax = BigDecimal.ZERO;
            } else {
                BigDecimal calTaxPart = base.subtract(Constants.TAX_STAGE_FREE_BASE);
                wxTax = calTaxPart.multiply(taxRate);
            }
            partnerSettle.setWxTax(wxTax);
            wxIncome = wxIncome.add(base).subtract(wxTax);
            partnerSettle.setWxIncome(wxIncome);
            partnerSettle.setBankIncome(BigDecimal.ZERO);
            partnerSettle.setBankTax(BigDecimal.ZERO);
            return true;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            return false;
        }
    }

//    public boolean runSummary(String dateStr, int userId) {
//        try {
//            /**
//             * partner_account_log 跑结算日期
//             */
//            Date dat = DateUtils.parse(dateStr, "yyyy-MM-dd");
//            int yeartime = Integer.parseInt(DateUtils.getYear(dat));
//            int monthtime = Integer.parseInt(DateUtils.getMonth(dat));
//            int daytime = Integer.parseInt(DateUtils.format(dat, "dd"));
//            PartnerAccountLog partnerAccountLog = partnerAccountLogService.getOneByUidAndDay(userId, yeartime, monthtime, daytime);
//            /**
//             * partner_income_summary 结算的数据月份 比dat早一个月
//             */
//            long beginTs = DateUtils.getLastMonth1stTimestamp(dat.getTime()) / 1000L;
//            long endTs  = DateUtils.getMonthLastTimestamp(dat.getTime()) / 1000L;
//            Date dat2 = new Date(beginTs  * 1000L);
//            int yeartime2 = Integer.parseInt(DateUtils.getYear(dat2));
//            int monthtime2 = Integer.parseInt(DateUtils.getMonth(dat2));
//            PartnerIncomeSummary partnerIncomeSummary = partnerIncomeSummaryService.getOneByUidYearMonth(userId, yeartime2, monthtime2);
//
//            /**
//             * 获取显示的汇总数据
//             */
//            PartnerSummaryContent partnerSummaryContent = partnerRewardService.getPartnerSummaryContentByUidAndTime(userId, beginTs, endTs);
//            /**
//             * 补差额
//             */
//            BigDecimal sum = partnerSummaryContent.getCash()
//                    .add(partnerSummaryContent.getUpCash())
//                    .add(partnerSummaryContent.getIcash())
//                    .add(partnerSummaryContent.getPcash())
//                    .add(partnerSummaryContent.getScash())
//                    .add(partnerSummaryContent.getRcash());
//            BigDecimal delta = partnerAccountLog.getChangeNum().subtract(sum);
//            partnerSummaryContent.setRcash(partnerSummaryContent.getRcash().add(delta));
//            /**
//             * 显示精度
//             */
//            partnerSummaryContent.setCash(partnerSummaryContent.getCash().setScale(2, RoundingMode.HALF_UP));
//            partnerSummaryContent.setUpCash(partnerSummaryContent.getUpCash().setScale(2, RoundingMode.HALF_UP));
//            partnerSummaryContent.setIcash(partnerSummaryContent.getIcash().setScale(2, RoundingMode.HALF_UP));
//            partnerSummaryContent.setPcash(partnerSummaryContent.getPcash().setScale(2, RoundingMode.HALF_UP));
//            partnerSummaryContent.setScash(partnerSummaryContent.getScash().setScale(2, RoundingMode.HALF_UP));
//            partnerSummaryContent.setRcash(partnerSummaryContent.getRcash().setScale(2, RoundingMode.HALF_UP));
//            /**
//             * 更新数据
//             */
//            partnerIncomeSummary.setContent(JSON.toJSONString(partnerSummaryContent));
//            partnerIncomeSummary.setIncome(partnerAccountLog.getChangeNum());
//            partnerIncomeSummaryService.updateById(partnerIncomeSummary);
//            return true;
//        } catch (Exception e) {
//            logger.error(e.getMessage(), e);
//            return false;
//        }
//    }
}
