package com.lanren.huhu.partner.schedule;

import com.lanren.huhu.partner.domain.AgentRevokeRecords;
import com.lanren.huhu.partner.domain.AgentReward;
import com.lanren.huhu.partner.service.AgentRevokeRecordsService;
import com.lanren.huhu.partner.service.AgentRewardService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

import static com.lanren.huhu.partner.constants.Constants.*;

/**
 * @author chen
 * @title: RevokeAgentRewardTask
 * @projectName partner
 * @description: 扣回代理商的奖励
 * @package com.lanren.huhu.partner.schedule
 * @date 2019-07-04 11:41
 */
@Component
public class RevokeAgentRewardTask {
    private static Logger logger = LoggerFactory.getLogger(RevokeAgentRewardTask.class);

    @Autowired
    AgentRevokeRecordsService agentRevokeRecordsService;
    @Autowired
    AgentRewardService agentRewardService;
    @Autowired
    DataSourceTransactionManager dataSourceTransactionManager;
    @Autowired
    TransactionDefinition transactionDefinition;

//    @Scheduled(fixedDelay = 5000L)
    public void runScheduledTask() {
        logger.info("run RevokeAgentRewardTask");
        runConsume();
    }

    private void runConsume() {
        List<AgentRevokeRecords> revokeRecordList = null;
        try {
            revokeRecordList = agentRevokeRecordsService.getAllPending();
            if (null == revokeRecordList || revokeRecordList.size() == 0) {
                return;
            }
            process(revokeRecordList);
        } catch (Exception e) {
            logger.error("process revoke failed. {}", revokeRecordList);
            logger.error(e.getMessage(), e);
        }
    }

    @Async
    public void process(List<AgentRevokeRecords> revokeRecordList) {
        for (AgentRevokeRecords agentRevokeRecord : revokeRecordList) {
            process(agentRevokeRecord);
        }
    }

    private void process(AgentRevokeRecords revokeRecord) {
        TransactionStatus transactionStatus = null;
        try {
            int revokeType = revokeRecord.getRevokeType();
            if (!AGENT_REVOKE_TYPE_MAP.keySet().contains(revokeType)) {
                logger.info("未知的奖励类型 跳过不处理: {}", revokeRecord);
                return;
            }
            /**
             * 开启事务
             */
            transactionStatus = dataSourceTransactionManager.getTransaction(transactionDefinition);
            List<AgentReward> agentRewardList = null;
            List<AgentReward> insertList = new ArrayList<AgentReward>();
            List<AgentReward> updateList = new ArrayList<AgentReward>();

            if (revokeType == AGENT_REVOKE_TYPE_ACTIVATION || revokeType == AGENT_REVOKE_TYPE_SELF_ORDER) {
                agentRewardList = agentRewardService.getListByRefId(revokeRecord.getRefId());
            } else if (revokeType == AGENT_REVOKE_TYPE_SHARE_ORDER) {
                agentRewardList = agentRewardService.getListBySubOrderSn(revokeRecord.getOrderType(), revokeRecord.getOrderSn(), revokeRecord.getSubOrderSn());
            } else {
                revokeRecord.setState(REVOKE_STATE_DO_NOTHING);
            }
            /**
             * 已结算状态 插负, 未结算状态 抹平
             */
            if (null != agentRewardList && agentRewardList.size() > 0) {
                for (AgentReward agentReward : agentRewardList) {
                    if (agentReward.getSettleState() == SETTLE_STATE_DONE) {
                        agentReward.setAmount(agentReward.getAmount().negate());
                        agentReward.setCommission(agentReward.getCommission().negate());
                        insertList.add(agentReward);
                    } else if (agentReward.getSettleState() == SETTLE_STATE_ON_THE_WAY || agentReward.getSettleState() == SETTLE_STATE_PRE) {
                        agentReward.setAmount(new BigDecimal(0));
                        agentReward.setCommission(new BigDecimal(0));
                        updateList.add(agentReward);
                    }
                }
                if (updateList.size() > 0) {
                    agentRewardService.updateBatch(updateList);
                }
                if (insertList.size() > 0) {
                    agentRewardService.batchInsert(insertList);
                }
                revokeRecord.setState(REVOKE_STATE_DONE);
            } else {
                revokeRecord.setState(REVOKE_STATE_DO_NOTHING);
            }
            agentRevokeRecordsService.updateById(revokeRecord);
            /**
             * 提交事务
             */
            dataSourceTransactionManager.commit(transactionStatus);
        } catch (Exception e) {
            /**
             * 回滚事务
             */
            dataSourceTransactionManager.rollback(transactionStatus);
            logger.error(e.getMessage(), e);
            revokeRecord.setState(REVOKE_STATE_FAILED);
            agentRevokeRecordsService.updateById(revokeRecord);
        }
    }
}
