package com.lanren.huhu.partner.schedule;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.lanren.huhu.partner.domain.AgentReward;
import com.lanren.huhu.partner.domain.UserAgent;
import com.lanren.huhu.partner.model.AgentRewardMessage;
import com.lanren.huhu.partner.model.ParentAgent;
import com.lanren.huhu.partner.model.User;
import com.lanren.huhu.partner.service.AgentRewardService;
import com.lanren.huhu.partner.service.PartnerAccountService;
import com.lanren.huhu.partner.service.UserAgentService;
import com.lanren.huhu.partner.service.UserService;
import com.lanren.huhu.partner.util.LockUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.ListOperations;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.scheduling.annotation.Async;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import static com.lanren.huhu.partner.constants.Constants.*;

/**
 * @author chen
 * @title: AgentRewardQueueTask
 * @projectName partner
 * @description: 消费 代理商奖励的消息队列, 处理代理商奖励, 写到agent_reward
 * @package com.lanren.huhu.partner.schedule
 * @date 2019-06-29 15:39
 */
@Component
public class AgentRewardQueueTask {
    private static Logger logger = LoggerFactory.getLogger(AgentRewardQueueTask.class);

    @Autowired
    StringRedisTemplate stringRedisTemplate;
    @Autowired
    UserAgentService userAgentService;
    @Autowired
    UserService userService;
    @Autowired
    AgentRewardService agentRewardService;
    @Autowired
    PartnerAccountService partnerAccountService;

    @Scheduled(fixedDelay = 5000L)
    public void runScheduledTask() {
        logger.info("run AgentRewardQueueTask");
        runConsume();
        logger.info("run AgentRewardQueueTask done");
    }

    public void runConsume() {
        ListOperations<String, String> ops = stringRedisTemplate.opsForList();
        while(null != ops && null != ops.size(AGENT_REWARD_QUEUE_KEY) && ops.size(AGENT_REWARD_QUEUE_KEY) > 0L) {
            String msg = "";
            try {
                /**
                 * 加分布式锁
                 */
                LockUtil.lock(DISTRIBUTE_REDIS_LOCK_KEY);
                try {
                    msg = ops.rightPop(AGENT_REWARD_QUEUE_KEY);
                } catch (Exception e) {
                    logger.error(e.getMessage(), e);
                    continue;
                } finally{
                    LockUtil.unlock(DISTRIBUTE_REDIS_LOCK_KEY);
                }

                logger.info("msg is {}", msg);
                JSONObject json = JSON.parseObject(msg);
                AgentRewardMessage message = json.toJavaObject(AgentRewardMessage.class);
                processReward(message);
            } catch (Exception e) {
                logger.error("process agent reward message failed", msg);
                logger.error(e.getMessage(), e);
            }
        }
    }

    @Async
    public void processReward(AgentRewardMessage message) {
        ListOperations<String, String> ops = stringRedisTemplate.opsForList();
        logger.info("异步处理 agent reward: {}", message);
        int rewardType = message.getRewardType();
        Map<Integer, BigDecimal> rateMap;
        if (!AGENT_RATE_MAP.keySet().contains(rewardType)) {
            logger.error("未知的奖励类型 跳过不处理: {}", message);
            return;
        } else {
            rateMap = AGENT_RATE_MAP.get(rewardType);
        }
        UserAgent userAgent = userAgentService.getOneByAgentId(message.getAgentId());
        if (userAgent == null) {
            logger.error("代理商id: {} 不存在, 跳过不处理", message);
            return;
        }
        if (userAgent.getAgentLevel() != AGENT_LEVEL_4) {
            logger.error("代理商id: {} 不是城市代理, 跳过不处理", message);
            return;
        }
        User user = userService.getRelationByUserId(userAgent.getUserId());
        ArrayList<ParentAgent> parentList = user.getAgentList();
        ArrayList<ParentAgent> level4List = new ArrayList<ParentAgent>();
        if (parentList.size() > 0) {
            /**
             * 扫描agentlist中的平推城市代理
             * 最多只需要找3个
             */
            int rewardCnt = 0;
            ArrayList<ParentAgent> rewardList = new ArrayList<ParentAgent>();
            for (ParentAgent agent : parentList) {
                /**
                 * 先扫描平推代理, 写到level4List
                 * 如果不是平推, 奖励次数还不够3, 写到rewardList
                 */
                if (agent.getLevel() == AGENT_LEVEL_4 && rewardCnt < 3) {
                    level4List.add(agent);
                    rewardList.add(agent);
                    rewardCnt++;
                } else if (rewardCnt < 3){
                    rewardList.add(agent);
                } else {
                    break;
                }
            }
            /**
             * 如果平推层级小于3, 又没有上级代理, 报错不处理
             */
            if (level4List.size() < 3 && level4List.size() == rewardList.size()) {
                logger.error("代理商关系错误, 只有平推, 没有上级. {}", message);
                return;
            }
            /**
             * 如果上面检查通过, 证明代理商关系没问题, 后面处理只看rewardList的size
             * 决定比例怎么分: 1个人-拿3份; 2个人-第二个拿后2份; 3个人-各1份
             */
            List<BigDecimal> rateList = new ArrayList<BigDecimal>();
            if (rewardList.size() == 1) {
                rateList.add(rateMap.get(0).add(rateMap.get(1)).add(rateMap.get(2)));
            } else if (rewardList.size() == 2) {
                rateList.add(rateMap.get(0));
                rateList.add(rateMap.get(1).add(rateMap.get(2)));
            } else if (rewardList.size() == 3) {
                rateList.add(rateMap.get(0));
                rateList.add(rateMap.get(1));
                rateList.add(rateMap.get(2));
            }
            for (int i=0; i<rewardList.size(); i++) {
                BigDecimal commissionRate = rateList.get(i);
                if (doInsert(commissionRate, rewardList.get(i), message)) {
                    logger.info("插入代理商分成奖励,奖励比例: {}, 代理商: {}", commissionRate, rewardList.get(i));
                } else {
                    logger.error("插入失败, 写回到队列");
//                    ops.leftPush(AGENT_REWARD_QUEUE_KEY, JSON.toJSONString(message));
                }
            }
        }
    }

    private boolean doInsert(BigDecimal commissionRate, ParentAgent agent, AgentRewardMessage message) {
        try {
            int partnerLevel = partnerAccountService.getOneByUserId(agent.getUserId()).getPartnerLevel();
            int sourceUserLevel = partnerAccountService.getOneByUserId(message.getSourceUserId()).getPartnerLevel();
            AgentReward agentReward = new AgentReward();
            agentReward.setRewardType(message.getRewardType());
            agentReward.setRechargeTime(LocalDateTime.parse(message.getRechargeTime(), DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));
            agentReward.setSourceUserId(message.getSourceUserId());
            agentReward.setReferenceId(Long.parseLong(message.getReferenceId()));
            agentReward.setAgentId(agent.getAgentId());
            agentReward.setCommissionAcount(message.getAgentReward().multiply(commissionRate));
            agentReward.setCommissionRate(commissionRate);
            agentReward.setSettleState(message.getSettleState());
            agentReward.setOrderType(message.getOrderType());
            agentReward.setOrderSn(message.getOrderSn());
            agentReward.setSubOrderSn(message.getSubOrderSn());
            agentReward.setMoney(message.getOrderCommission());
            agentReward.setAgentLevel(agent.getLevel());
            agentReward.setUserId(agent.getUserId());
            agentReward.setRewardRemark("测试");
            agentReward.setAmount(message.getAgentReward().multiply(commissionRate));
            agentReward.setSourceOrder(message.getReferenceId());
            agentReward.setSourceUserLevel(sourceUserLevel);
            agentReward.setCreatedAt(System.currentTimeMillis() / 1000L);
            if (agentRewardService.save(agentReward)) {
                return true;
            } else {
                return false;
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            return false;
        }
    }
}
